package com.mmc.oms.filter;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.mmc.oms.common.ResultBody;
import com.mmc.oms.common.ResultEnum;
import com.mmc.oms.common.Tenant;
import com.mmc.oms.config.Audience;
import com.mmc.oms.config.TenantContext;
import com.mmc.oms.model.dto.UserAccountDTO;
import io.jsonwebtoken.Claims;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import javax.servlet.*;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * @Author small
 * @Date 2023/5/28 17:02
 * @Version 1.0
 */
@Slf4j
@Component
@WebFilter(filterName = "AuthSignatureFilter", urlPatterns = "/*")
public class AuthSignatureFilter implements AuthFilter {

    @Autowired
    private Audience audience;


    @Autowired
    private StringRedisTemplate stringRedisTemplate;
    /**
     * 无需登录白名单
     */
    private static final String[] IGNORE_URLS = {"/oms/swagger/swagger-resources","/oms/swagger/v2/api-docs","/oms/swagger/doc.html"};

    /*无需加密狗无需登录白名单*/
    private static final String[] USE_KEY = {"/crm/account/loginByUsbKey"};

    /**
     * 请求方式预请求方式值
     */

    private static final String REQUEST_METHOD_OPTIONS_VALUE = "OPTIONS";

    public static final String SWAGGER_URL_PREFIX = "/oms/swagger";

    @Override
    public void init(FilterConfig filterConfig) {
        log.info(" filter name is 'AuthSignatureFilter' init success");
    }

    /**
     * 过滤器前置处理
     *
     * @param request
     * @param response
     * @return
     */
    @Override
    public boolean before(HttpServletRequest request, HttpServletResponse response) {
        String url = request.getRequestURI();
        // 忽略以下url请求,白名单路径以及swagger路径
        if (!ArrayUtils.contains(IGNORE_URLS, url)
                && !ArrayUtils.contains(USE_KEY, url) && !url.startsWith("/oms/webjars") && !url.startsWith(SWAGGER_URL_PREFIX)) {
            if (REQUEST_METHOD_OPTIONS_VALUE.equals(request.getMethod())) {
                response.setStatus(HttpServletResponse.SC_OK);
                return false;
            } else {
                String token = request.getHeader("token");
                if (null==token){
                    response(response,ResultBody.error(ResultEnum.THE_REQUEST_IS_NOT_AUTHENTICATED));
                    return false;
                }
                String s = stringRedisTemplate.opsForValue().get(token);
                if (null==s){
                    response(response,ResultBody.error(ResultEnum.THE_TOKEN_IS_INVALID));
                    return false;
                }
                UserAccountDTO userAccountDTO = JSON.parseObject(s, UserAccountDTO.class);
                try {
                    Tenant tenant = TenantContext.buildTenant(userAccountDTO.getAccountNo());
                    if (tenant!=null){
                        TenantContext.setTenant(tenant);
                        return true;
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
        return true;
    }

    @Override
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain)
            throws IOException, ServletException {
        final HttpServletRequest request = (HttpServletRequest) req;
        final HttpServletResponse response = (HttpServletResponse) res;

        if (!before(request, response)) {
            return;
        }
        chain.doFilter(req, res);
        after();
    }

    /**
     * 过滤器后置处理
     *
     * @return
     */
    @Override
    public boolean after() {
        TenantContext.clear(); // 上下文清理
        return true;
    }

    private static final String APPLICATION_JSON_UTF8 = "application/json;charset=UTF-8";

    public static void response(HttpServletResponse response, ResultBody resultBody) {
        response.setContentType(APPLICATION_JSON_UTF8);
        try {
            response
                    .getWriter()
                    .write(JSON.toJSONString(ResultBody.error(resultBody.getCode(), resultBody.getMessage())));
        } catch (IOException e) {
            log.error(e.getMessage());
        }
    }

    private static String toString(Object obj) {
        if (null == obj) {
            return "";
        }
        return obj.toString();
    }

    @Override
    public void destroy() {
    }
}
