package com.mmc.oms.service.uav.impl;

import com.alibaba.fastjson2.JSONObject;
import com.mmc.oms.client.PmsClient;
import com.mmc.oms.common.ResultEnum;
import com.mmc.oms.common.constant.UavOrderProportion;
import com.mmc.oms.common.json.JsonUtil;
import com.mmc.oms.common.result.PageResult;
import com.mmc.oms.common.result.ResultBody;
import com.mmc.oms.common.util.CodeUtil;
import com.mmc.oms.common.util.KdnExpressUtil;
import com.mmc.oms.dao.UserAddressDao;
import com.mmc.oms.dao.uav.UavOrderDao;
import com.mmc.oms.dao.uav.UavOrderExpressDao;
import com.mmc.oms.entity.mall.UserAddressDO;
import com.mmc.oms.entity.uav.*;
import com.mmc.oms.enums.UavOrderStatus;
import com.mmc.oms.feign.PaymentAppApi;
import com.mmc.oms.feign.UserAppApi;
import com.mmc.oms.model.dto.kdn.KdnExpDTO;
import com.mmc.oms.model.dto.order.ExpStationsDTO;
import com.mmc.oms.model.dto.uav.*;
import com.mmc.oms.model.dto.user.BaseAccountDTO;
import com.mmc.oms.model.qo.uav.UavOrderQO;
import com.mmc.oms.model.vo.ApplyRefundVO;
import com.mmc.oms.model.vo.demand.CommonPaymentVO;
import com.mmc.oms.model.vo.uav.*;
import com.mmc.oms.model.vo.wallet.TopUpOrderVO;
import com.mmc.oms.service.demand.DemandReleaseOrderService;
import com.mmc.oms.service.uav.UavOrderService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author: zj
 * @Date: 2023/9/4 15:31
 */
@Service
@Slf4j
public class UavOrderServiceImpl implements UavOrderService {
    @Autowired
    private UavOrderDao uavOrderDao;

    @Autowired
    private UavOrderExpressDao uavOrderExpressDao;

    @Autowired
    private UserAddressDao userAddressDao;

    @Autowired
    private DemandReleaseOrderService demandReleaseOrderService;

    @Autowired
    private UserAppApi userAppApi;

    @Autowired
    private PmsClient pmsClient;

    @Autowired
    private PaymentAppApi paymentAppApi;

    @Override
    @Transactional
    public ResultBody addOrder(AddUavOrderVO param, BaseAccountDTO currentAccount) {
        if (CollectionUtils.isEmpty(param.getSkuVOS())){
            return ResultBody.error("至少选择一个规格！");
        }
        MallGoodsVO mallGoodsVO = null;
        try {
            mallGoodsVO = pmsClient.mallGoodsDetails(param.getMallGoodsId());
            // 检验商品是否在卖
            if (mallGoodsVO == null || !mallGoodsVO.getShelfStatus().equals(1)){
                return ResultBody.error("商品已下架");
            }
            // 找到对应的规格
            for (UavOrderSkuVO skuVO : param.getSkuVOS()) {
                Boolean flag = false;
                for (PriceStockVO priceStockVO : mallGoodsVO.getPriceStock()) {
                    if (skuVO.getProductSpec().equals(priceStockVO.getProductSpec())) {
                        // 获取最新的规格单价
                        skuVO.setUnitPrice(priceStockVO.getSalePrice());
                        flag = true;
                        break;
                    }
                }
                if (!flag){
                    return ResultBody.error("选择规格不存在");
                }
            }
        }catch (Exception e){
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            e.printStackTrace();
        }
        String orderNo = CodeUtil.uavOrderCode();
        BigDecimal orderTotalAmount = BigDecimal.ZERO;
        BigDecimal otherAmount = BigDecimal.ZERO;
        for (UavOrderSkuVO skuVO : param.getSkuVOS()) {
            // 计算规格金额
            BigDecimal skuAmount = BigDecimal.valueOf(skuVO.getOrderNum()).multiply(skuVO.getUnitPrice());
            orderTotalAmount = orderTotalAmount.add(skuAmount);
        }
        otherAmount = otherAmount.add(orderTotalAmount);
        // 校验云享金、钱包余额是否足够抵扣
        BigDecimal shareAmount = BigDecimal.ZERO;
        BigDecimal salaryAmount = BigDecimal.ZERO;
        ResultBody<PayWalletDTO> payResInfo = userAppApi.getCurrentUserPayWalletInfo(currentAccount.getToken());

        if (!payResInfo.getCode().equals(ResultEnum.SUCCESS.getResultCode()) || payResInfo.getResult() == null){
            return payResInfo;
        }
        PayWalletDTO payWalletDTO = payResInfo.getResult();
        log.info("payWalletDTO:{}", JSONObject.toJSONString(payWalletDTO));
        if (param.getDeductShareAmount().equals(1)){
            if (otherAmount.compareTo(payWalletDTO.getCashAmt()) >= 0) {
                // 抵扣云享金小于等于订单总额时，直接使用云享金
                shareAmount = payWalletDTO.getCashAmt();
            }else {
                // 余额超过订单金额时，订单总额使用余额支付
                shareAmount = otherAmount;
            }
        }
        otherAmount = otherAmount.subtract(shareAmount);
        Integer statusCode = null;
        log.info("otherAmount:{}，shareAmount：{}", otherAmount,shareAmount);
        // 支付完成
        if (otherAmount.compareTo(BigDecimal.ZERO) ==0 ){
            statusCode = UavOrderStatus.WAITING_DELIVER_GOODS.getCode();
        }else {
            if (param.getDeductSalaryAmount().equals(1)){
                if (otherAmount.compareTo(payWalletDTO.getSalaryAmt()) >= 0) {
                    // 抵扣余额小于等于订单总额时，直接使用余额
                    salaryAmount = payWalletDTO.getSalaryAmt();
                }else {
                    // 余额超过订单金额时，订单总额使用余额支付
                    salaryAmount = otherAmount;
                }
            }
            otherAmount = otherAmount.subtract(salaryAmount);
            if (otherAmount.compareTo(BigDecimal.ZERO) == 0) {
                statusCode = UavOrderStatus.WAITING_DELIVER_GOODS.getCode();
            }
        }
        PayUavWalletVO payUavWalletVO = new PayUavWalletVO();
        payUavWalletVO.setCashAmount(shareAmount);
        payUavWalletVO.setSalaryAmount(salaryAmount);
        payUavWalletVO.setOrderStatus(100);
        payUavWalletVO.setUserAccountId(currentAccount.getUserAccountId());
        ResultBody resultBody = userAppApi.feignPayUavWallet(payUavWalletVO, currentAccount.getToken());
        if (!resultBody.getCode().equals(ResultEnum.SUCCESS.getResultCode())){
            return resultBody;
        }
        // 添加总订单表
        UavOrderDO uavOrderDO = new UavOrderDO();
        uavOrderDO.setOrderNo(orderNo);
        uavOrderDO.setOrderTotalAmount(orderTotalAmount);
        uavOrderDO.setOrderType(param.getOrderType());
        // TODO 意向订单待处理
        uavOrderDO.setStatusCode(otherAmount.compareTo(BigDecimal.ZERO) > 0 ? UavOrderStatus.PAYING.getCode() : statusCode);
        uavOrderDO.setUserAccountId(currentAccount.getUserAccountId());
        uavOrderDO.setThirdBackUserAccountId(param.getThirdBackUserAccountId());
        uavOrderDO.setCompanyName(param.getCompanyName());
        uavOrderDO.setSalaryAmount(salaryAmount);
        uavOrderDO.setShareAmount(shareAmount);
        uavOrderDO.setOtherAmount(otherAmount);
        uavOrderDO.setUserAddressId(param.getUserAddressId());
        uavOrderDO.setUserRemark(param.getUserRemark());
        uavOrderDao.addOrder(uavOrderDO);

        List<UavOrderSkuDO> skuDOS = new ArrayList<>();
        for (UavOrderSkuVO skuVO : param.getSkuVOS()) {
            UavOrderSkuDO uavOrderSkuDO = new UavOrderSkuDO();
            // 添加订单规格信息
            uavOrderSkuDO.setUavOrderId(uavOrderDO.getId());
            uavOrderSkuDO.setMallGoodsId(param.getMallGoodsId());
            uavOrderSkuDO.setTradeName(mallGoodsVO.getTradeName());
            uavOrderSkuDO.setPriceStockId(skuVO.getPriceStockId());
            uavOrderSkuDO.setProductSpec(skuVO.getProductSpec());
            uavOrderSkuDO.setOrderNum(skuVO.getOrderNum());
            uavOrderSkuDO.setUnitPrice(skuVO.getSalePrice());
            uavOrderSkuDO.setSkuNo(skuVO.getSkuNo());
            uavOrderSkuDO.setSkuImage(skuVO.getSkuImage());
            skuDOS.add(uavOrderSkuDO);
        }
        uavOrderDao.addOrderSku(skuDOS);
        // 添加收货地址
        UserAddressDO userAddressDO = userAddressDao.selectById(param.getUserAddressId());
        UavOrderExpressDO uavOrderExpressDO = new UavOrderExpressDO(userAddressDO);
        uavOrderExpressDO.setUavOrderId(uavOrderDO.getId());
        uavOrderExpressDao.addAddress(uavOrderExpressDO);
        return ResultBody.success(uavOrderDO.buildUavOrderDTO());
    }

    @Override
    public ResultBody detail(Integer id) throws Exception {
        UavOrderDO uavOrderDO = uavOrderDao.detail(id);
        UavOrderDTO uavOrderDTO = uavOrderDO.buildUavOrderDTO();
        // 收货信息、付款信息、规格信息、快递信息、签署信息
        UavOrderExpressDO uavOrderExpressDO = uavOrderExpressDao.getUavOrderExpressDO(id);
        if (uavOrderExpressDO != null) {
            uavOrderDTO.setUavOrderExpressDTO(uavOrderExpressDO.buildUavOrderExpressDTO());
        }
        List<UavOrderPayDO> payDOS = uavOrderDao.listUavOrderPay(id);
        if (CollectionUtils.isNotEmpty(payDOS)) {
            List<UavOrderPayDTO> collect = payDOS.stream().map(UavOrderPayDO::buildUavOrderPayDTO).collect(Collectors.toList());
            uavOrderDTO.setPayDTOList(collect);
        }
        List<UavOrderSkuDO> skuDOS =uavOrderDao.listUavOrderSkuDO(id);
        if (CollectionUtils.isNotEmpty(skuDOS)) {
            List<UavOrderSkuDTO> uavOrderSkuDTOS = skuDOS.stream().map(UavOrderSkuDO::buildUavOrderSkuDTO).collect(Collectors.toList());
            uavOrderDTO.setSkuDTOList(uavOrderSkuDTOS);
        }
        // 快递信息
        KdnExpDTO kdnExpDTO = this.getOrderExpInfo(uavOrderExpressDO);
        if (kdnExpDTO != null) {
            uavOrderDTO.setKdnExpDTO(kdnExpDTO);
        }
        return ResultBody.success(uavOrderDTO);
    }

    private KdnExpDTO getOrderExpInfo(UavOrderExpressDO exp) throws Exception {
        String json = "";
        if ("SF".equals(exp.getSendExpCode())) {
            json = KdnExpressUtil.SF(exp.getTakePhone(), exp.getSendExpNo());
        } else if ("JD".equals(exp.getSendExpCode())) {
            json = KdnExpressUtil.JD("", exp.getSendExpNo());
        } else {
            json = KdnExpressUtil.others(exp.getSendExpCode(), exp.getSendExpNo());
        }
        KdnExpDTO kdn = JsonUtil.parseJsonToObj(json, KdnExpDTO.class);
        kdn.setLogisticCode(exp.getSendExpNo());
        if (!org.springframework.util.CollectionUtils.isEmpty(kdn.getTraces())) {
            Collections.sort(kdn.getTraces(), Comparator.comparing(ExpStationsDTO::getAcceptTime).reversed());// 升序
        }
        return kdn;
    }

    @Transactional
    @Override
    public ResultBody close(Integer id, String token) {
        UavOrderDO uavOrderDO = uavOrderDao.detail(id);
        // 修改订单状态
        this.updateUavOrderStatus(id, UavOrderStatus.CLOSE);
        // 退云享金、佣金
        PayUavWalletVO payUavWalletVO = new PayUavWalletVO();
        payUavWalletVO.setCashAmount(uavOrderDO.getShareAmount());
        payUavWalletVO.setSalaryAmount(uavOrderDO.getSalaryAmount());
        payUavWalletVO.setOrderStatus(1400);
        payUavWalletVO.setUserAccountId(uavOrderDO.getUserAccountId());
        ResultBody resultBody = userAppApi.feignPayUavWallet(payUavWalletVO, token);
        if (!resultBody.getCode().equals(ResultEnum.SUCCESS.getResultCode())){
            return resultBody;
        }
        // 微信退款
        ApplyRefundVO applyRefundVO = new ApplyRefundVO();
        applyRefundVO.setOutTradeNo(uavOrderDO.getOrderNo());
        applyRefundVO.setReason("关闭订单");
        applyRefundVO.setRefund(uavOrderDO.getOtherAmount().multiply(BigDecimal.valueOf(100)).longValue());
        ResultBody applyRefund = paymentAppApi.applyRefund(applyRefundVO);
        if (!applyRefund.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody list(UavOrderQO uavOrderQO, BaseAccountDTO currentAccount) {
        if (!currentAccount.getRoleInfo().getSuperAdmin().equals(1)){
            uavOrderQO.setThirdBackUserAccountId(currentAccount.getUserAccountId());
        }
        int count = uavOrderDao.countList(uavOrderQO);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(uavOrderQO.getPageNo(), uavOrderQO.getPageSize(), count));
        }
        Integer pageNo = uavOrderQO.getPageNo();
        uavOrderQO.buildCurrentPage();
        List<UavOrderDO> uavOrderDOS = uavOrderDao.list(uavOrderQO);
        List<UavOrderDTO> uavOrderDTOS = uavOrderDOS.stream().map(UavOrderDO::buildUavOrderDTO).collect(Collectors.toList());
        return ResultBody.success(PageResult.buildPage(pageNo, uavOrderQO.getPageSize(), count, uavOrderDTOS));
    }

    @Override
    public ResultBody listByUid(UavOrderQO uavOrderQO, BaseAccountDTO currentAccount) {
        uavOrderQO.setUserAccountId(currentAccount.getUserAccountId());
        int count = uavOrderDao.countList(uavOrderQO);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(uavOrderQO.getPageNo(), uavOrderQO.getPageSize(), count));
        }
        Integer pageNo = uavOrderQO.getPageNo();
        uavOrderQO.buildCurrentPage();
        List<UavOrderDO> uavOrderDOS = uavOrderDao.list(uavOrderQO);
        List<UavOrderDTO> uavOrderDTOS = uavOrderDOS.stream().map(UavOrderDO::buildUavOrderDTO).collect(Collectors.toList());
        return ResultBody.success(PageResult.buildPage(pageNo, uavOrderQO.getPageSize(), count, uavOrderDTOS));
    }

    @Override
    public ResultBody pay(Integer id, BaseAccountDTO currentAccount, HttpServletRequest request) {
        // 发起微信支付
        UavOrderDO uavOrderDO = uavOrderDao.detail(id);
        if (!uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.PAYING.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        CommonPaymentVO commonPaymentVO = new CommonPaymentVO();
        int shouldPay = uavOrderDO.getOtherAmount().multiply(BigDecimal.valueOf(100)).intValue();
        if (shouldPay <= 0) {
            return ResultBody.success("已支付完成");
        }
        List<UavOrderSkuDO> orderSkuDOS = uavOrderDao.listUavOrderSkuDO(id);
        StringBuffer sb = new StringBuffer();
        orderSkuDOS.stream().forEach(uavOrderSkuDO -> sb.append(uavOrderSkuDO.getProductSpec()));
        commonPaymentVO.setAmount(shouldPay);
        commonPaymentVO.setDescription(sb.toString());
        commonPaymentVO.setOrderNumber(uavOrderDO.getOrderNo());
        commonPaymentVO.setOrderPort(0);
        ResultBody resultBody = demandReleaseOrderService.orderPayment(commonPaymentVO, request, currentAccount.getUserAccountId());
        return resultBody;
    }

    @Transactional
    @Override
    public ResultBody send(UavOrderExpressVO param) {
        // 修改订单状态
        UavOrderDO uavOrderDO = uavOrderDao.detail(param.getUavOrderId());
        if (!uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.WAITING_DELIVER_GOODS.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        this.updateUavOrderStatus(param.getUavOrderId(), UavOrderStatus.WAITING_DELIVER_GOODS);
        // 修改收货信息
        UavOrderExpressDO uavOrderExpressDO = new UavOrderExpressDO(param);
        uavOrderExpressDao.updateUavOrderExpressDO(uavOrderExpressDO);
        return ResultBody.success();
    }

    @Transactional
    @Override
    public ResultBody receive(Integer id, BaseAccountDTO currentAccount) {
        // 修改订单状态
        UavOrderDO uavOrderDO = uavOrderDao.detail(id);
        if (!uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.WAITING_RECEIVE_GOODS.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        this.updateUavOrderStatus(id, UavOrderStatus.WAITING_RECEIVE_GOODS);
        // 修改收货信息
        UavOrderExpressDO uavOrderExpressDO = uavOrderExpressDao.getUavOrderExpressDO(id);
        uavOrderExpressDO.setReceive(1);
        uavOrderExpressDO.setReceiveTime(new Date());
        uavOrderExpressDao.updateUavOrderExpressDO(uavOrderExpressDO);
        if (uavOrderDO.getOrderType().equals(0)) {
            // 订单抽成设置
            BigDecimal thirdAmount = uavOrderDO.getOrderTotalAmount().multiply(
                    BigDecimal.valueOf(100).subtract(UavOrderProportion.PROPORTION).divide(BigDecimal.valueOf(100)));
            PayUavWalletVO payUavWalletVO = new PayUavWalletVO();
            payUavWalletVO.setCashAmount(BigDecimal.ZERO);
            payUavWalletVO.setSalaryAmount(thirdAmount);
            payUavWalletVO.setOrderStatus(1500);
            payUavWalletVO.setRemark("商城订单收入，订单编号" + uavOrderDO.getOrderNo());
            // 拿到商家管理员的用户id
            ResultBody managerRes = userAppApi.getManagerIdByBackUserId(uavOrderDO.getThirdBackUserAccountId(), currentAccount.getToken());
            if (!managerRes.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
                return managerRes;
            }
            payUavWalletVO.setUserAccountId((Integer) managerRes.getResult());
            ResultBody resultBody = userAppApi.feignPayUavWallet(payUavWalletVO, currentAccount.getToken());
            if (!resultBody.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
                return resultBody;
            }
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody remarkOrder(UavOrderRemarkVO uavOrderRemarkVO) {
        // 填入评价内容
        // 修改订单状态
        return null;
    }

    @Transactional
    public void updateUavOrderStatus(Integer id, UavOrderStatus uavOrderStatus){
        uavOrderDao.updateUavOrderStatus(id, uavOrderStatus.getNextCode());
    }

    @Override
    public ResultBody sellerRemark(Integer id, String content) {
        uavOrderDao.sellerRemark(id, content);
        return ResultBody.success();
    }

    @Override
    public ResultBody statusList() {
        List<UavOrderStatusDO> statusDOS = uavOrderDao.statusList();
        List<UavOrderStatusDTO> uavOrderStatusDTOS = statusDOS.stream().map(UavOrderStatusDO::buildUavOrderStatusDTO).collect(Collectors.toList());
        return ResultBody.success(uavOrderStatusDTOS);
    }

    @Override
    public ResultBody upLoadPay(UavOrderPayVO uavOrderPayVO) {
        return null;
    }

    @Override
    public ResultBody checkPay(UavOrderPayVO uavOrderPayVO) {
        return null;
    }

    @Transactional
    @Override
    public void payUavOrder(TopUpOrderVO topUpOrderVO) {
        // 添加支付凭证、记录支付方式
        UavOrderDO uavOrderDO = uavOrderDao.detailByNo(topUpOrderVO.getOrderNo());
        if (uavOrderDO == null) {
            throw new RuntimeException();
        }
        this.updateUavOrderStatus(uavOrderDO.getId(), UavOrderStatus.WAITING_DELIVER_GOODS);
        // 修改订单信息
        uavOrderDO.setPayType("wechatpay");
        uavOrderDO.setPayTime(new Date());
        uavOrderDao.updateUavOrderPayInfo(uavOrderDO);
        UavOrderPayDO uavOrderPayDO = new UavOrderPayDO();
        uavOrderPayDO.setUavOrderId(uavOrderDO.getId());
        uavOrderPayDO.setPayRemark(topUpOrderVO.getTradeStateDesc());
        uavOrderPayDO.setPayType(0);
        uavOrderPayDO.setPayAmount(BigDecimal.valueOf(topUpOrderVO.getAmount()).divide(BigDecimal.valueOf(100)));
        uavOrderPayDO.setCheckStatus(1);
        uavOrderPayDO.setCreateTime(new Date());
        uavOrderDao.addOrderPay(uavOrderPayDO);
    }
}
