package com.mmc.oms.service.lease.impl;

import com.mmc.oms.common.ResultEnum;
import com.mmc.oms.common.result.ResultBody;
import com.mmc.oms.common.util.CodeUtil;
import com.mmc.oms.dao.UserAddressDao;
import com.mmc.oms.dao.lease.LeaseOrderDao;
import com.mmc.oms.entity.lease.LeaseOrderDO;
import com.mmc.oms.entity.lease.LeaseOrderExpressDO;
import com.mmc.oms.entity.mall.UserAddressDO;
import com.mmc.oms.enums.LeaseOrderStatus;
import com.mmc.oms.enums.LeaseTermInfo;
import com.mmc.oms.feign.PmsAppApi;
import com.mmc.oms.feign.UserAppApi;
import com.mmc.oms.model.dto.lease.LeaseOrderDTO;
import com.mmc.oms.model.dto.uav.PayWalletDTO;
import com.mmc.oms.model.dto.user.BaseAccountDTO;
import com.mmc.oms.model.vo.lease.AddLeaseOrderVO;
import com.mmc.oms.model.vo.lease.LeaseGoodsVO;
import com.mmc.oms.model.vo.lease.LeasePriceStockVO;
import com.mmc.oms.model.vo.uav.GoodsResourcesVO;
import com.mmc.oms.service.lease.LeaseOrderService;
import com.mmc.oms.service.uav.UavOrderService;
import com.mmc.oms.util.TDateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Date;

/**
 * @Author LW
 * @date 2023/9/21 16:46
 * 概要：
 */
@Service
@Slf4j
public class LeaseOrderServiceImpl implements LeaseOrderService {
    @Resource
    private PmsAppApi pmsAppApi;
    @Resource
    private UavOrderService uavOrderService;
    @Resource
    private LeaseOrderDao leaseOrderDao;
    @Resource
    private UserAddressDao userAddressDao;
    @Resource
    private UserAppApi userAppApi;

    @Override
    @Transactional
    public ResultBody addLeaseOrder(AddLeaseOrderVO param, BaseAccountDTO currentAccount) {
        log.info("user:{},addOrder param:{}", currentAccount.getUserAccountId(), param);
        if (StringUtils.isEmpty(param.getSku())) {
            return ResultBody.error("至少选择一个规格！");
        }
        if (param.getWareNum() <= 0) {
            return ResultBody.error("购买数量不得少于1");
        }
        LeaseGoodsVO leaseGoodsVO = null;
        LeasePriceStockVO priceStock = null;
        try {
            leaseGoodsVO = pmsAppApi.leaseGoodsDetails(param.getLeaseGoodsId()).getResult();
            // 检验商品是否在卖
            if (leaseGoodsVO == null || !leaseGoodsVO.getShelfStatus().equals(1)) {
                return ResultBody.error("商品已下架");
            }
            // 判断租赁时间是否超出当前商品的最长租期或小于最短租期
            int distanceDays = TDateUtil.distanceDays(param.getStartDate(), param.getEndDate());
            param.setLeaseTerm(distanceDays);
            // 获取租赁商品的最低租期
            Integer minLeaseTerm = getLeaseTerm(leaseGoodsVO.getMinLeaseTerm());
            // 获取租赁商品的最大租期
            Integer maxLeaseTerm = getLeaseTerm(leaseGoodsVO.getMaxLeaseTerm());
            if (distanceDays < minLeaseTerm || distanceDays > maxLeaseTerm) {
                return ResultBody.error("商品信息发生变更，刷新后重试!");
            }
            // 判断该租期在那个价格区间
            Integer priceSection = getPriceSection(distanceDays);
            // 找到对应的规格
            boolean flag = false;
            for (LeasePriceStockVO priceStockVO : leaseGoodsVO.getPriceStock()) {
                if (param.getSku().equals(priceStockVO.getProductSpec())) {
                    priceStock = priceStockVO;
                    // 获取最新的规格单价
                    param.setUnitPrice(getUnitPrice(priceSection, priceStockVO));
                    param.setDeposit(priceStockVO.getCashPledge());
                    flag = true;
                    break;
                }
            }
            if (!flag) {
                return ResultBody.error("选择规格不存在");
            }
        } catch (Exception e) {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            return ResultBody.error(e.getMessage());
        }

        String orderNo = CodeUtil.generateOrderId();
        // 计算订单总金额
        BigDecimal orderTotalAmount = param.getUnitPrice().multiply(BigDecimal.valueOf(param.getLeaseTerm()))
                .multiply(BigDecimal.valueOf(param.getWareNum())).add(param.getDeposit().multiply(BigDecimal.valueOf(param.getWareNum())));
        BigDecimal otherAmount = BigDecimal.ZERO;
        otherAmount = otherAmount.add(orderTotalAmount);
        // 判断云享金或余额是否够抵扣
        // 校验云享金、钱包余额是否足够抵扣
        BigDecimal shareAmount = BigDecimal.ZERO;
        BigDecimal salaryAmount = BigDecimal.ZERO;
        Integer statusCode = null;
        // 获取用户钱包信息
        ResultBody<PayWalletDTO> payResInfo = userAppApi.getCurrentUserPayWalletInfo(currentAccount.getToken());
        if (!payResInfo.getCode().equals(ResultEnum.SUCCESS.getResultCode()) || payResInfo.getResult() == null) {
            return payResInfo;
        }
        PayWalletDTO payWalletDTO = payResInfo.getResult();
        if (param.getDeductShareAmount().equals(1)) {
            if (otherAmount.compareTo(payWalletDTO.getCashAmt()) >= 0) {
                // 抵扣云享金小于等于订单总额时，直接使用云享金
                shareAmount = payWalletDTO.getCashAmt();
            } else {
                // 余额超过订单金额时，订单总额使用余额支付
                shareAmount = otherAmount;
            }
        }
        otherAmount = otherAmount.subtract(shareAmount);
        // 支付完成
        if (otherAmount.compareTo(BigDecimal.ZERO) == 0) {
            statusCode = LeaseOrderStatus.PAID.getCode();
        } else {
            if (param.getDeductSalaryAmount().equals(1)) {
                if (otherAmount.compareTo(payWalletDTO.getSalaryAmt()) >= 0) {
                    // 抵扣余额小于等于订单总额时，直接使用余额
                    salaryAmount = payWalletDTO.getSalaryAmt();
                } else {
                    // 余额超过订单金额时，订单总额使用余额支付
                    salaryAmount = otherAmount;
                }
            }
            otherAmount = otherAmount.subtract(salaryAmount);
            if (otherAmount.compareTo(BigDecimal.ZERO) == 0) {
                statusCode = LeaseOrderStatus.PAID.getCode();
            }
        }
        // 抵扣余额
        uavOrderService.deductWallet(shareAmount, salaryAmount, currentAccount);
        // 获取商品主图
        String url = null;
        for (GoodsResourcesVO goodsResourcesVO : leaseGoodsVO.getResourcesList()) {
            if (goodsResourcesVO.getType().equals(0)) {
                url = goodsResourcesVO.getUrl();
                break;
            }
        }
        // 添加总订单表
        LeaseOrderDO leaseOrderDO = new LeaseOrderDO();
        leaseOrderDO.setOrderNo(orderNo);
        leaseOrderDO.setOrderTotalAmount(orderTotalAmount);
        leaseOrderDO.setStatusCode(otherAmount.compareTo(BigDecimal.ZERO) > 0 ? LeaseOrderStatus.PLACED_AN_ORDER.getCode() : statusCode);
        leaseOrderDO.setUserAccountId(currentAccount.getUserAccountId());
        leaseOrderDO.setLeaseGoodsId(param.getLeaseGoodsId());
        leaseOrderDO.setCompanyName(param.getCompanyName());
        leaseOrderDO.setThirdBackUserAccountId(param.getThirdBackUserAccountId());
        leaseOrderDO.setOrderName(leaseGoodsVO.getTradeName());
        if (priceStock.getSkuImage() != null) {
            leaseOrderDO.setSkuImg(priceStock.getSkuImage());
        } else {
            leaseOrderDO.setSkuImg(url);
        }
        leaseOrderDO.setSalaryAmount(salaryAmount);
        leaseOrderDO.setSku(param.getSku());
        leaseOrderDO.setShareAmount(shareAmount);
        leaseOrderDO.setOtherAmount(otherAmount);
        leaseOrderDO.setUserAddressId(param.getUserAddressId());
        leaseOrderDO.setUserRemark(param.getUserRemark());
        leaseOrderDO.setUnitPrice(param.getUnitPrice());
        leaseOrderDO.setWareNum(param.getWareNum());
        leaseOrderDO.setDeposit(param.getDeposit());
        leaseOrderDO.setStartDate(param.getStartDate());
        if (null != statusCode && statusCode.equals(LeaseOrderStatus.PAID.getCode())) {
            leaseOrderDO.setPayTime(new Date());
        }
        leaseOrderDO.setEndDate(param.getEndDate());
        leaseOrderDao.addOrder(leaseOrderDO);
        // 添加收货地址
        UserAddressDO userAddressDO = userAddressDao.selectById(param.getUserAddressId());
        LeaseOrderExpressDO leaseOrderExpressDO = new LeaseOrderExpressDO(userAddressDO);
        leaseOrderExpressDO.setLeaseOrderId(leaseOrderDO.getId());
        leaseOrderDao.addAddress(leaseOrderExpressDO);
        LeaseOrderDTO leaseOrderDTO = leaseOrderDO.buildLeaseOrderDTO();
        leaseOrderDTO.setModeOfDeliveryInfo(leaseGoodsVO.getModeOfDeliveryInfo());
        return ResultBody.success(leaseOrderDTO);
    }

    private Integer getLeaseTerm(Integer leaseTermId) {
        if (leaseTermId.equals(LeaseTermInfo.THREE_DAY.getId())) {
            return LeaseTermInfo.THREE_DAY.getDays();
        } else if (leaseTermId.equals(LeaseTermInfo.SEVEN_DAY.getId())) {
            return LeaseTermInfo.SEVEN_DAY.getDays();
        } else if (leaseTermId.equals(LeaseTermInfo.THIRTY_DAY.getId())) {
            return LeaseTermInfo.THIRTY_DAY.getDays();
        } else if (leaseTermId.equals(LeaseTermInfo.NINETY_DAY.getId())) {
            return LeaseTermInfo.NINETY_DAY.getDays();
        } else {
            return LeaseTermInfo.ONE_HUNDRED_AND_EIGHTY_DAY.getDays();
        }
    }


    private BigDecimal getUnitPrice(Integer priceSection, LeasePriceStockVO priceStockVO) {
        if (priceSection.equals(LeaseTermInfo.THREE_DAY.getDays())) {
            return priceStockVO.getThreeDaysRental();
        } else if (priceSection.equals(LeaseTermInfo.SEVEN_DAY.getDays())) {
            return priceStockVO.getSevenDaysRental();
        } else if (priceSection.equals(LeaseTermInfo.THIRTY_DAY.getDays())) {
            return priceStockVO.getThirtyDaysRental();
        } else if (priceSection.equals(LeaseTermInfo.NINETY_DAY.getDays())) {
            return priceStockVO.getNinetyDaysRental();
        } else {
            return priceStockVO.getMaxDaysRental();
        }
    }

    private Integer getPriceSection(int distanceDays) {
        if (distanceDays >= LeaseTermInfo.THREE_DAY.getDays() && distanceDays < LeaseTermInfo.SEVEN_DAY.getDays()) {
            return LeaseTermInfo.THREE_DAY.getDays();
        } else if (distanceDays >= LeaseTermInfo.SEVEN_DAY.getDays() && distanceDays < LeaseTermInfo.THIRTY_DAY.getDays()) {
            return LeaseTermInfo.SEVEN_DAY.getDays();
        } else if (distanceDays >= LeaseTermInfo.THIRTY_DAY.getDays() && distanceDays < LeaseTermInfo.NINETY_DAY.getDays()) {
            return LeaseTermInfo.THIRTY_DAY.getDays();
        } else if (distanceDays >= LeaseTermInfo.NINETY_DAY.getDays() && distanceDays < LeaseTermInfo.ONE_HUNDRED_AND_EIGHTY_DAY.getDays()) {
            return LeaseTermInfo.NINETY_DAY.getDays();
        } else {
            return LeaseTermInfo.ONE_HUNDRED_AND_EIGHTY_DAY.getDays();
        }
    }
}
