package com.mmc.csf.release.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.mmc.csf.common.util.date.TDateUtil;
import com.mmc.csf.common.util.math.CodeUtil;
import com.mmc.csf.common.util.page.PageResult;
import com.mmc.csf.common.util.web.ResultBody;
import com.mmc.csf.common.util.web.ResultEnum;
import com.mmc.csf.infomation.dto.GetOrderNumberDTO;
import com.mmc.csf.infomation.dto.PayWalletDTO;
import com.mmc.csf.release.dao.CurriculumOrderDao;
import com.mmc.csf.release.dao.FlyerTrainingDao;
import com.mmc.csf.release.entity.curriculum.*;
import com.mmc.csf.release.feign.UserAppApi;
import com.mmc.csf.release.flyer.qo.CurriculumQo;
import com.mmc.csf.release.flyer.vo.*;
import com.mmc.csf.release.service.FlyerTrainingService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;


/**
 * @Author LW
 * @date 2023/5/15 10:11
 * 概要：
 */
@Slf4j
@Service
@Transactional(rollbackFor = Exception.class)
public class FlyerTrainingServiceImpl implements FlyerTrainingService {

    @Resource
    private FlyerTrainingDao flyerTrainingDao;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    @Autowired
    private RestTemplate restTemplate;

    @Value("${iuav.userapp.url}")
    private String userApp;

    @Qualifier("com.mmc.csf.release.feign.UserAppApi")
    @Autowired
    private UserAppApi userAppApi;

    @Resource
    private CurriculumOrderDao curriculumOrderDao;

    @Override
    public ResultBody pilotRegistration(RegistrationVO registrationVO, Integer userAccountId) {
        PilotRegistrationDO pilotRegistrationDO = new PilotRegistrationDO(registrationVO);
        pilotRegistrationDO.setApplyUserId(userAccountId);
        if (CollectionUtils.isNotEmpty(registrationVO.getIndustryAppAuth())) {
            // 将行业应用认证的id用逗号拼接
            String industryAppAuth = registrationVO.getIndustryAppAuth().stream().map(Object::toString).collect(Collectors.joining(","));
            pilotRegistrationDO.setIndustryAppAuth(industryAppAuth);
        }
        flyerTrainingDao.insertPilotRegistrationInfo(pilotRegistrationDO);
        return ResultBody.success();
    }

    @Override
    public ResultBody getDronePilotLicense() {
        List<DronePilotLicenseVO> allLicenses = flyerTrainingDao.selectAllLicenses();
        List<DronePilotLicenseVO> topLevelLicenses = new ArrayList<>();
        Map<Integer, DronePilotLicenseVO> licenseMap = new HashMap<>();

        // 将每个数据模型对象添加到Map中，以便在递归过程中查找它们的父母
        for (DronePilotLicenseVO license : allLicenses) {
            license.setChildLicenses(new ArrayList<>());
            licenseMap.put(license.getId(), license);
        }

        // 构建树结构
        for (DronePilotLicenseVO license : allLicenses) {
            if (license.getParentId() == 0) {
                topLevelLicenses.add(license);
            } else {
                DronePilotLicenseVO parent = licenseMap.get(license.getParentId());
                parent.getChildLicenses().add(license);
            }
        }
        return ResultBody.success(topLevelLicenses);
    }

    @Override
    public ResultBody getIndustryFlightSkills() {
        List<FlightSkillsDO> flightSkillsList = flyerTrainingDao.listIndustryFlightSkills();
        return ResultBody.success(flightSkillsList.stream()
                .map(FlightSkillsDO::buildFlightSkillsVO).collect(Collectors.toList()));
    }

    @Override
    public PageResult queryCurriculumInfoList(CurriculumQo curriculumQo) {
        int count = flyerTrainingDao.countCurriculumInfoList(curriculumQo);
        if (count == 0) {
            return PageResult.buildPage(curriculumQo.getPageNo(), curriculumQo.getPageSize(), count);
        }
        Integer pageNo = curriculumQo.getPageNo();
        curriculumQo.buildCurrentPage();
        List<CurriculumInfoDO> curriculumList = flyerTrainingDao.listPageCurriculumInfo(curriculumQo);
        List<CurriculumInfoVO> list = curriculumList.stream().map(CurriculumInfoDO::buildCurriculumInfoVO).collect(Collectors.toList());
        return PageResult.buildPage(pageNo, curriculumQo.getPageSize(), count, list);
    }


    @Override
    public ResultBody licenseOrSkillQueryCurriculum(Integer type) {
        List<CurriculumInfoDO> curriculumInfoList;
        if (type == 0) {
            // 表示查询考证的
            curriculumInfoList = flyerTrainingDao.selectLicensesCurriculum();
        } else {
            curriculumInfoList = flyerTrainingDao.selectSkillCurriculum(type);
        }
        return ResultBody.success(curriculumInfoList == null ? null : curriculumInfoList.stream()
                .map(CurriculumInfoDO::buildCurriculumInfoVO).collect(Collectors.toList()));
    }

    @Override
    public ResultBody curriculumDetails(Integer id) {
        CurriculumInfoDO curriculumInfoDO = flyerTrainingDao.curriculumDetails(id);
        if (curriculumInfoDO == null) {
            return ResultBody.error("课程不存在或已下架！");
        } else {
            CurriculumInfoVO curriculumInfoVO = curriculumInfoDO.buildCurriculumInfoVO();
            return ResultBody.success(curriculumInfoVO);
        }
    }

    @Override
    public Integer addCurriculum(CurriculumInfoVO curriculumInfoVO) {
        CurriculumInfoDO currInfoDO = new CurriculumInfoDO(curriculumInfoVO);
        return flyerTrainingDao.addCurriculum(currInfoDO);
    }

    @Override
    public Integer updateCurriculum(CurriculumInfoVO curriculumInfoVO) {
        CurriculumInfoDO currInfoDO = new CurriculumInfoDO(curriculumInfoVO);
        return flyerTrainingDao.updateCurriculum(currInfoDO);
    }

    @Override
    public Integer removeCurriculum(Integer id) {
        return flyerTrainingDao.removeCurriculum(id);
    }

    @Override
    public ResultBody treeCurriculumSkill() {
        List<FlightSkillsDO> flightSkillsList = flyerTrainingDao.treeCurriculumSkill();
        List<FlightSkillsVO> collect = flightSkillsList.stream().map(FlightSkillsDO::buildFlightSkillsVO).collect(Collectors.toList());
        return ResultBody.success(collect);
    }

    @Override
    public List<CurriculumInfoVO> feignCurriculumInfoList(Integer categoriesId) {
        CurriculumQo curriculumQo = new CurriculumQo();
        curriculumQo.setPageNo(1);
        curriculumQo.setPageSize(Integer.MAX_VALUE);
        int count = flyerTrainingDao.countCurriculumInfoList(curriculumQo);
        if (count == 0) {
            return null;
        }
        Integer pageNo = curriculumQo.getPageNo();
        curriculumQo.buildCurrentPage();
        List<CurriculumInfoDO> curriculumList = flyerTrainingDao.listPageCurriculumInfo(curriculumQo);
        List<CurriculumInfoVO> list = curriculumList.stream().map(CurriculumInfoDO::buildCurriculumInfoVO).collect(Collectors.toList());
        return list;
    }

    @Override
    public int feignCountCurriculumInfo(Integer categoriesId) {
        int count = flyerTrainingDao.countCurriculumInfo(categoriesId);
        return count;
    }

    @Override
    public Integer getCurriculumCount() {
        return flyerTrainingDao.getCurriculumCount();
    }

    @Override
    public Integer addCurriculumClassify(CurriculumClassifyVO curriculumClassify) {
        CurriculumClassifyDO curriculumClassifyDO = new CurriculumClassifyDO(curriculumClassify);
        Integer count;
        if(curriculumClassifyDO.getOneCourseId() == null){
            count = flyerTrainingDao.countOneCourseClassify();
            curriculumClassifyDO.setOneCourseId(count + 1);
        } else {
            count = flyerTrainingDao.countTwoCourseClassify(curriculumClassify.getOneCourseId());
            curriculumClassifyDO.setTwoCourseId(count);
        }
        return flyerTrainingDao.addCurriculumClassify(curriculumClassifyDO);
    }

    @Override
    public Integer updateCurriculumClassify(CurriculumClassifyVO curriculumClassify) {
        CurriculumClassifyDO curriculumClassifyDO = new CurriculumClassifyDO(curriculumClassify);
        return flyerTrainingDao.updateCurriculumClassify(curriculumClassifyDO);
    }

    @Override
    public ResultBody removeCurriculumClassify(Integer id) {
        CurriculumClassifyDO classifyDO = flyerTrainingDao.selectCurriculumClassifyById(id);
        if(classifyDO.getTwoCourseId() == null && flyerTrainingDao.countTwoCourseClassify(classifyDO.getOneCourseId() ) > 1){
           return ResultBody.error("有二级分类无法删除");
        }
        flyerTrainingDao.removeCurriculumClassify(id);
        return ResultBody.success();
    }

    @Override
    public ResultBody selectCurriculumClassify() {
        flyerTrainingDao.selectCurriculumClassify();
        List<CurriculumClassifyDO> curriculumClassifyList = flyerTrainingDao.selectCurriculumClassify();
        List<CurriculumClassifyVO> collect = curriculumClassifyList.stream().map(CurriculumClassifyDO::buildCurriculumClassifyVO).collect(Collectors.toList());
        return ResultBody.success(collect);
    }

    @Override
    public CurriculumInfoDO selectCurriculumInfoById(Integer id) {
        return flyerTrainingDao.selectCurriculumInfoById(id);
    }

    @Override
    public Integer curriculumPayByIntegral(CurriculumOrderDO curriculumOrderDO) {
        return flyerTrainingDao.insertCurriculumOrder(curriculumOrderDO);
    }

    @Override
    public ResultBody curriculumPayByMoney(HttpServletRequest request,CurriculumPayVO curriculumPayVO,Integer accountId) {
        //获取当前课程的内容
        CurriculumInfoDO curriculumInfoDO = flyerTrainingDao.selectCurriculumInfoById(curriculumPayVO.getId());
        // 获取支付类型
        String[] split = curriculumPayVO.getPaymentType().split(",");
        Set<String> collect = Arrays.asList(split).stream().collect(Collectors.toSet());
        TreeSet<String> objects = new TreeSet<>(collect);

        // 微信已经支付的金额
        BigDecimal weChatPay = new BigDecimal(0);
        if (objects.contains("3")) {
            // 获取当前微信已经支付的金额
            String json = stringRedisTemplate.opsForValue().get(curriculumPayVO.getPaymentOrderNumber() + accountId);
            GetOrderNumberDTO orderNumberDTO = JSONObject.parseObject(json, GetOrderNumberDTO.class);
            weChatPay = orderNumberDTO.getWeChatPay();
        }

        //获取用户钱包
        ResultBody resultBody = getCurrentUserPayWalletInfo(request);
        PayWalletDTO payWalletDTO = (PayWalletDTO) resultBody.getResult();

        //用户剩余的佣金
        BigDecimal userSalaryAmt = new BigDecimal(0);
        userSalaryAmt = payWalletDTO.getSalaryAmt();
        //云享金余额还剩余多少
        BigDecimal userCashAmt = new BigDecimal(0);
        userCashAmt = payWalletDTO.getCashAmt();
        // 需要支付的总金额(除去微信已经支付过的金额，还需要支付的金额)
        BigDecimal tempTotalAmount = new BigDecimal(0);
        tempTotalAmount = curriculumInfoDO.getRequireAmout().subtract(weChatPay);

        CurriculumOrderDO curriculumOrderDO = new CurriculumOrderDO();
        curriculumOrderDO.setUserAccountId(accountId);
        curriculumOrderDO.setCurriculumInfoId(curriculumInfoDO.getId());
        // 生成支付流水号(规则"CO" + 时间戳 + 用户id + 随机数4位)
        curriculumOrderDO.setOrderNo("CO" + TDateUtil.getDateStr(new Date(), "yyyyMMddHHmmss") + curriculumInfoDO.getId() + CodeUtil.getRandomNum(4));

        if(userSalaryAmt.add(userCashAmt).compareTo(tempTotalAmount) < 0 ){
            if(!(weChatPay.compareTo(BigDecimal.ZERO) == 0)){
//                log("云享金或者佣金不够支付,支付失败,请回滚微信支付中的金额:" + curriculumPayVO.getPaymentOrderNumber());
                return ResultBody.error("云享金或者佣金不够支付,支付失败,请回滚微信支付中的金额");
            }else{
                return  ResultBody.error("云享金或者佣金不够支付,支付失败");
            }
        }
        CurriculumOrderLogVO curriculumOrderLogVO = new CurriculumOrderLogVO();
        curriculumOrderLogVO.setOrderNo(curriculumOrderDO.getOrderNo());
        curriculumOrderLogVO.setUserAccountId(accountId);
        if(objects.contains("1")){
            if(tempTotalAmount.compareTo(userCashAmt) <= 0){
                curriculumOrderLogVO.setCashAmtPaid(tempTotalAmount);
                userCashAmt = tempTotalAmount;
                userSalaryAmt = BigDecimal.ZERO;
                tempTotalAmount = BigDecimal.ZERO;
            }else{
                curriculumOrderLogVO.setCashAmtPaid(userCashAmt);
                tempTotalAmount = tempTotalAmount.subtract(userCashAmt);
            }
        }
        if(!objects.contains("2")&& tempTotalAmount.compareTo(BigDecimal.ZERO) > 0){
            return  ResultBody.error("云享金不够支付,支付失败");
        }
        if(objects.contains("2") && tempTotalAmount.compareTo(BigDecimal.ZERO) >= 0) {
            if(tempTotalAmount.compareTo(userSalaryAmt) <= 0){
                curriculumOrderLogVO.setSalaryAmtPaid(tempTotalAmount);
                userSalaryAmt = tempTotalAmount;
            }else{
                return  ResultBody.error("佣金不够支付,支付失败");
            }
        }
        userAppApi.feignCurriculumOrderLog(request.getHeader("token"),request,curriculumOrderLogVO);
        curriculumOrderDO.setOrderTotalAmount(curriculumInfoDO.getRequireAmout());
        curriculumOrderDO.setSalaryAmount(userSalaryAmt);
        curriculumOrderDO.setWeChat(weChatPay);
        curriculumOrderDO.setCashAmount(userCashAmt);
        flyerTrainingDao.insertCurriculumOrder(curriculumOrderDO);
        return  ResultBody.success("支付成功");
    }

    /**
     * 获取用户钱包信息
     *
     * @param
     * @param request
     * @return
     */
    public ResultBody getCurrentUserPayWalletInfo(HttpServletRequest request) {
        String token = request.getHeader("token");
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("token", token);
        FlyerTrainingServiceImpl.log.info("获取钱包信息的token:{}",token);
        HttpEntity<MultiValueMap<String, Object>> formEntity = new HttpEntity<MultiValueMap<String, Object>>(headers);
        ResponseEntity<String> exchange = null;
        try {
            exchange = restTemplate.exchange(userApp + "/userapp/pay/getCurrentUserPayWalletInfo", HttpMethod.GET, formEntity, String.class);
        } catch (RestClientException e) {
            return ResultBody.error(ResultEnum.THE_THIRD_PARTY_INTERFACE_IS_BEING_UPDATED);
        }
        Object body = exchange.getBody();
        JSONObject jsonObject = JSONObject.parseObject((String) body);
        JSONObject result1 = (JSONObject) jsonObject.get("result");
        PayWalletDTO payWalletDTO = JSON.parseObject(result1.toJSONString(), PayWalletDTO.class);
        return ResultBody.success(payWalletDTO);
    }

    /**
     * 分页查询订单列表
     * @param param
     * @return {@link List}<{@link CurriculumOrderDO}>
     */
    @Override
    public ResultBody getCurriculumOrderList(CurriculumOrderQO param) {
        int count = curriculumOrderDao.getCountCurriculumOrder(param);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(param.getPageNo(), param.getPageSize(), count));
        }
        int pageNo = param.getPageNo();
        param.buildCurrentPage();
        List<CurriculumOrderDO> pageList = curriculumOrderDao.selectOrderList(param);

        return ResultBody.success(PageResult.buildPage(pageNo, param.getPageSize(), count, pageList));
    }
    @Override
    public ResultBody deleteCurriculumOrderInfo(Integer id) {
        curriculumOrderDao.deleteByPrimaryKey(id);
        return ResultBody.success();
    }

}
