package com.mmc.csf.release.service.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.mmc.csf.common.util.page.PageResult;
import com.mmc.csf.common.util.web.ResultBody;
import com.mmc.csf.infomation.dto.UserAccountSimpleDTO;
import com.mmc.csf.release.auth.qo.UserAccountQO;
import com.mmc.csf.release.constant.TokenConstant;
import com.mmc.csf.release.dao.CommentDAO;
import com.mmc.csf.release.dao.DynamicDAO;
import com.mmc.csf.release.dao.LikeDAO;
import com.mmc.csf.release.entity.forum.ForumDynamicDO;
import com.mmc.csf.release.entity.forum.ForumResourceDO;
import com.mmc.csf.release.feign.UserAppApi;
import com.mmc.csf.release.forum.vo.CommentVO;
import com.mmc.csf.release.forum.vo.DynamicVO;
import com.mmc.csf.release.forum.vo.MediaVO;
import com.mmc.csf.release.forum.vo.UserBaseInfoVO;
import com.mmc.csf.release.service.DynamicService;

/**
 * @Author LW
 *
 * @date 2023/5/15 10:29 概要：动态信息实现层
 */
@Service
public class DynamicServiceImpl implements DynamicService {

  @Resource private DynamicDAO dynamicDAO;
  @Resource private LikeDAO likeDAO;
  @Resource private CommentDAO commentDAO;
  @Autowired private UserAppApi userAppApi;

  @Override
  @Transactional(rollbackFor = Exception.class)
  public ResultBody insertDynamic(DynamicVO dynamicVO, Integer userAccountId) {
    // 插入动态基本信息
    ForumDynamicDO forumDynamicDO = new ForumDynamicDO(dynamicVO);
    forumDynamicDO.setRootPath(null);
    forumDynamicDO.setUserAccountId(userAccountId);
    dynamicDAO.insertDynamic(forumDynamicDO);
    // 插入动态资源
    if (CollectionUtils.isNotEmpty(dynamicVO.getMediaVO())) {
      for (MediaVO mediaVO : dynamicVO.getMediaVO()) {
        dynamicDAO.insertResource(forumDynamicDO.getId(), mediaVO.getType(), mediaVO.getUrl());
      }
    }
    return ResultBody.success("动态发布成功");
  }

  @Override
  public ResultBody likeDynamic(Integer userId, Integer dynamicId) {
    // 获取动态信息
    ForumDynamicDO forumDynamicDO = dynamicDAO.getDynamicInfo(dynamicId);
    if (!likeDAO.isLiked(userId, dynamicId)) {
      likeDAO.insertLike(userId, dynamicId);
      dynamicDAO.updateDynamicLikeCount(dynamicId, forumDynamicDO.getVersion());
    } else {
      likeDAO.deleteLike(userId, dynamicId);
      dynamicDAO.updateSubDynamicLikeCount(dynamicId, forumDynamicDO.getVersion());
    }
    return ResultBody.success();
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public ResultBody commentDynamic(CommentVO commentVO, Integer userAccountId) {
    ForumDynamicDO forumDynamicDO = dynamicDAO.getDynamicInfo(commentVO.getDynamicId());
    if (forumDynamicDO == null) {
      return ResultBody.error("动态不存在或已删除");
    }
    // 新增评论
    ForumDynamicDO comment = new ForumDynamicDO();
    comment.setUserAccountId(userAccountId);
    comment.setDescription(commentVO.getContent());
    comment.defaultInfo();
    comment.setRootPath(commentVO.getRootPath());
    dynamicDAO.insertDynamic(comment);
    // 修改评论数量
    dynamicDAO.updateDynamicCommentCount(commentVO.getDynamicId(), forumDynamicDO.getVersion());
    return ResultBody.success();
  }

  @Override
  public ResultBody deleteComment(Integer id) {
    commentDAO.deleteComment(id);
    return ResultBody.success();
  }

  @Override
  public ResultBody dynamicDetails(Integer dynamicId, Integer userId, HttpServletRequest request) {
    // 判断此动态是否还存在
    //ForumDynamicDO dynamicInfo = dynamicDAO.getDynamicInfo(dynamicId);
    ForumDynamicDO dynamicInfo = dynamicDAO.getDynamicInfoById(dynamicId);
    if (dynamicInfo == null) {
      return ResultBody.error("动态不存在！");
    }
    DynamicVO dynamicVO = dynamicInfo.buildDynamicVO();
    // 设置是否点赞
    boolean liked = likeDAO.isLiked(userId, dynamicId);
    UserAccountSimpleDTO userAccountSimpleDTO =
        userAppApi.feignGetUserSimpleInfo(
            dynamicInfo.getUserAccountId(), request.getHeader(TokenConstant.TOKEN));
    UserBaseInfoVO userBaseInfoVO = new UserBaseInfoVO();
    userBaseInfoVO.setNickName(userAccountSimpleDTO.getNickName());
    userBaseInfoVO.setUserImg(userAccountSimpleDTO.getUserImg());
    dynamicVO.setUserBaseInfo(userBaseInfoVO);
    dynamicVO.setLikes(liked);
    // 获取媒体资源
    List<ForumResourceDO> forumResourceList = dynamicDAO.listDynamicResources(dynamicId);
    if (CollectionUtils.isNotEmpty(forumResourceList)) {
      List<MediaVO> MediaVOList =
          forumResourceList.stream().map(ForumResourceDO::buildMedia).collect(Collectors.toList());
      dynamicVO.setMediaVO(MediaVOList);
    }
    // 动态评论列表
    List<DynamicVO> listDynamic = getCommentInfo(dynamicId, request);
    dynamicVO.setCommentAndReplyVO(listDynamic);
    return ResultBody.success(dynamicVO);
  }

  private List<DynamicVO> getCommentInfo(Integer dynamicId, HttpServletRequest request) {
    // 获取下面所有的一级评论
    List<ForumDynamicDO> forumDynamic = dynamicDAO.getDynamicComment(dynamicId.toString());
    if (forumDynamic == null) {
      return new ArrayList<>();
    }
    List<Integer> userAccountIds =
        forumDynamic.stream().map(ForumDynamicDO::getUserAccountId).collect(Collectors.toList());
    UserAccountQO bUserAccountQO = new UserAccountQO();
    bUserAccountQO.setUserIds(userAccountIds);
    bUserAccountQO.setPageNo(1);
    bUserAccountQO.setPageSize(999);
    // 获取用户的信息
    List<UserAccountSimpleDTO> userAccountSimpleDTOS =
        userAppApi.feignListAppUserAccount(bUserAccountQO, request.getHeader(TokenConstant.TOKEN));

    List<DynamicVO> listDynamic =
        forumDynamic.stream().map(ForumDynamicDO::buildDynamicVO).collect(Collectors.toList());
    for (DynamicVO dynamicVO : listDynamic) {
      for (UserAccountSimpleDTO userAccountSimpleDTO : userAccountSimpleDTOS) {
        if (dynamicVO.getUserAccountId().equals(userAccountSimpleDTO.getId())) {
          UserBaseInfoVO userBaseInfoVO = new UserBaseInfoVO();
          userBaseInfoVO.setNickName(userAccountSimpleDTO.getNickName());
          userBaseInfoVO.setUserImg(userAccountSimpleDTO.getUserImg());
          userBaseInfoVO.setUserName(userAccountSimpleDTO.getUserName());
          userBaseInfoVO.setPhone(userAccountSimpleDTO.getPhoneNum());
          userBaseInfoVO.setUid(userAccountSimpleDTO.getUid());
          dynamicVO.setUserBaseInfo(userBaseInfoVO);
          break;
        }
      }
    }
    return listDynamic;
  }

  @Override
  public PageResult dynamicList(
      Integer pageNo, Integer pageSize, Integer userId, HttpServletRequest request) {
    int count = dynamicDAO.countDynamicList();
    if (count == 0) {
      return PageResult.buildPage(pageNo, pageSize, 0);
    }
    int itemIndex = (pageNo - 1) * pageSize;
    List<ForumDynamicDO> forumDynamicList = dynamicDAO.dynamicList(itemIndex, pageSize);
    // 动态id集合
    List<Integer> ids =
        forumDynamicList.stream().map(ForumDynamicDO::getId).collect(Collectors.toList());
    // 根据id集合查询图片
    List<ForumResourceDO> forumImageList = dynamicDAO.listDynamicPicture(ids);
    Map<Integer, List<ForumResourceDO>> ImageMap =
        forumImageList.stream().collect(Collectors.groupingBy(ForumResourceDO::getDynamicId));
    // 获取用户信息
    Set<Integer> userIds =
        forumDynamicList.stream().map(ForumDynamicDO::getUserAccountId).collect(Collectors.toSet());
    List<Integer> userIdList = new ArrayList<>(userIds);
    UserAccountQO bUserAccountQO = new UserAccountQO();
    bUserAccountQO.setUserIds(userIdList);
    List<UserAccountSimpleDTO> userAccountSimpleDTOS =
        userAppApi.feignListAppUserAccount(bUserAccountQO, request.getHeader(TokenConstant.TOKEN));
    Map<Integer, UserAccountSimpleDTO> userAccountInfoMap =
        userAccountSimpleDTOS.stream()
            .collect(Collectors.toMap(UserAccountSimpleDTO::getId, Function.identity()));
    // 获取liked信息
    List<ForumDynamicDO> forumDynamicDOS = likeDAO.selectLiked(userId, ids);
    List<DynamicVO> dynamicVOList =
        forumDynamicList.stream()
            .map(
                d -> {
                  DynamicVO dynamicVO = d.buildDynamicVO();
                  List<ForumResourceDO> forumResourceDOList = ImageMap.get(d.getId());
                  if (CollectionUtils.isNotEmpty(forumResourceDOList)) {
                    dynamicVO.setMediaVO(
                        forumResourceDOList.stream()
                            .map(ForumResourceDO::buildMedia)
                            .collect(Collectors.toList()));
                  }
                  UserAccountSimpleDTO userAccountSimpleDTO =
                      userAccountInfoMap.get(d.getUserAccountId());
                  if (userAccountSimpleDTO != null) {
                    UserBaseInfoVO userBaseInfoVO = new UserBaseInfoVO();
                    userBaseInfoVO.setNickName(userAccountSimpleDTO.getNickName());
                    userBaseInfoVO.setUserImg(userAccountSimpleDTO.getUserImg());
                    dynamicVO.setUserBaseInfo(userBaseInfoVO);
                  }
                  // 设置是否点赞
                  for (ForumDynamicDO forumDynamicDO : forumDynamicDOS) {
                    if (forumDynamicDO.getId().equals(d.getId())) {
                      dynamicVO.setLikes(true);
                      break;
                    } else {
                      dynamicVO.setLikes(false);
                    }
                  }
                  return dynamicVO;
                })
            .collect(Collectors.toList());
    return PageResult.buildPage(pageNo, pageSize, count, dynamicVOList);
  }
}
