import React, { useEffect, useState } from 'react';
import { RightOutlined } from '@ant-design/icons';
import { useRouter } from 'next/router';
import { useDispatch } from 'react-redux';
import styled from 'styled-components';
import { HomeAPI } from '@/api';
import { InterDataType } from '@/api/interface';
import { RecommendGoodsType } from '@/api/interface/home';
import { setGlobalData } from '@/store/module/globalData';

// 列表类型
type ListType = InterDataType<RecommendGoodsType>;

const HomeProductView: React.FC = () => {
  // 路由钩子
  const router = useRouter();
  // store
  const dispatch = useDispatch();
  // 跳转商品列表
  const handleMore = async () => {
    await router.push('/mall');
  };
  // 推荐商品列表
  const [recommendGoodsList, setRecommendGoodsList] = useState<
    ListType[0]['mallGoodsList']
  >([]);
  // 获取推荐商品
  const getRecommendGoodsList = async () => {
    const res = await HomeAPI.getRecommendGoods();
    if (res && res.code === '200') {
      const list = res.result || [];
      setRecommendGoodsList(
        list
          ?.map((i) => i.mallGoodsList)
          ?.flat()
          ?.slice(0, 6),
      );
    }
  };
  // 获取最低价格
  const getLowerPrice = (item: ListType[0]['mallGoodsList'][0]) => {
    return (
      item.priceStock?.reduce((a: any, b: any) =>
        a.salePrice < b.salePrice ? a : b,
      ).salePrice || 0
    );
  };
  // 跳转详情
  const handleDetail = (item: ListType[0]['mallGoodsList'][0]) => {
    dispatch(
      setGlobalData({
        loadingSpinnerVisible: true,
      }),
    );
    router.push(`/mall/product/${item.id}`).then();
  };
  // 组件挂载
  useEffect(() => {
    getRecommendGoodsList().then();
  }, []);
  return (
    <HomeProductWrap>
      <div className="product-head flex-between">
        <div className="label">大家都在买</div>
        <div className="action flex-end" onClick={handleMore}>
          <div className="text">更多</div>
          <RightOutlined />
        </div>
      </div>
      <div className="product-list flex-start">
        {recommendGoodsList?.map((i, j) => (
          <div
            className="list-item list-none"
            key={j}
            onClick={() => handleDetail(i)}
          >
            <img
              className="item-image"
              src={`${i.resourcesList[0].url}?x-oss-process=image/quality,q_50`}
              alt={i.tradeName}
            />
            <div className="item-title text-ellipsis">{i.tradeName}</div>
            <div className="item-price">
              <span className="label">￥</span>
              <span className="num">
                {i.priceShow ? getLowerPrice(i).toLocaleString() : '咨询报价'}
              </span>
            </div>
          </div>
        ))}
      </div>
    </HomeProductWrap>
  );
};

export default HomeProductView;
// 样式
const HomeProductWrap = styled.div`
  position: relative;
  width: 100%;
  box-sizing: border-box;
  margin-bottom: 1.5rem;
  .product-head {
    position: relative;
    width: 100%;
    box-sizing: border-box;
    margin-bottom: 1rem;
    .label {
      font-weight: 550;
      font-size: 1rem;
      color: #333333;
      user-select: none;
    }
    .action {
      color: #ff552d;
      cursor: pointer;
    }
  }
  .product-list {
    position: relative;
    width: 100%;
    box-sizing: border-box;
    .list-item {
      position: relative;
      width: calc((100% - 0.67rem * 5) / 6);
      background: #ffffff;
      border-radius: 0.44rem;
      border: 0.06rem solid #e3e3e3;
      overflow: hidden;
      padding: 0.67rem 0.33rem;
      box-sizing: border-box;
      cursor: pointer;
      &:not(:last-child) {
        margin-right: 0.67rem;
      }
      &:hover {
        filter: brightness(0.96);
      }
      .item-image {
        width: 100%;
        height: 7rem;
        object-fit: cover;
        margin-bottom: 0.33rem;
      }
      .item-title {
        width: 100%;
        font-weight: 550;
        font-size: 13px;
        color: #252525;
        line-height: 1rem;
        margin-bottom: 0.33rem;
      }
      .item-price {
        font-weight: 550;
        font-size: 13px;
        color: #ff1b1b;
        line-height: 1.04rem;
        .label {
          font-size: 12px;
        }
      }
    }
  }
`;
