import React from 'react';
import { useRouter } from 'next/router';
import styled from 'styled-components';
import { InterListType } from '@/api/interface';
import { QueryCurriculumInfoListType } from '@/api/interface/course';

// 列表类型
type DetailType = InterListType<QueryCurriculumInfoListType>[0];
// 课程类型列表
const courseAttributeList: { label: string; value: number; color: string }[] = [
  { label: '免费', value: 0, color: '#f99a0b' },
  { label: '积分', value: 1, color: '#00AD53' },
  { label: '付费', value: 2, color: '#FF4600' },
];

const CourseListItem: React.FC<{ detail: DetailType }> = ({ detail }) => {
  // 路由钩子
  const router = useRouter();
  // 获取当前课程属性
  const getCourseAttribute = () => {
    return courseAttributeList.find((i) => i.value === detail?.courseAttribute);
  };
  // 跳转商品详情
  const handleDetail = () => {
    router.push(`/course/detail/${detail?.id}`).then();
  };
  return (
    <CourseListItemWrap onClick={handleDetail}>
      <div className="mb-2 h-32 w-full" style={{ height: '10.5rem' }}>
        <img
          className="h-full w-full object-cover"
          src={
            detail?.surfaceUrl
              ? `${detail?.surfaceUrl}?x-oss-process=image/quality,q_25`
              : `${detail?.videoUrl}?x-oss-process=video/snapshot,t_1000,m_fast`
          }
          alt={detail?.curriculumName}
        />
      </div>
      <div className="course-content w-full px-2">
        <div className="content-title flex w-full flex-nowrap">
          <div className="title text-ellipsis">{detail?.curriculumName}</div>
          <div
            className="tag"
            style={{ background: getCourseAttribute()?.color }}
          >
            {getCourseAttribute()?.label}
          </div>
        </div>
        <div className="text-ellipsis text-xs text-gray-400">
          {detail?.curriculumDesc}
        </div>
        {!!detail?.requireAmout && (
          <div className="content-price">
            <span className="label">￥</span>
            <span className="num">{detail?.requireAmout}</span>
          </div>
        )}
        {!!detail?.requireIntegral && (
          <div className="content-price">
            <span className="num">{detail?.requireIntegral}</span>
            <span className="label">积分</span>
          </div>
        )}
      </div>
      <div className="course-footer flex justify-between">
        <div className="label">课程供应商</div>
        <div className="num">{(detail?.id || 1) * 333}人已学习</div>
      </div>
    </CourseListItemWrap>
  );
};

export default CourseListItem;
// 样式
const CourseListItemWrap = styled.div`
  position: relative;
  box-sizing: border-box;
  width: calc((100% - (0.83rem * 3)) / 4);
  height: 16.9rem;
  background: #ffffff;
  box-shadow: 0 0.17rem 0.86rem 0 rgba(65, 65, 65, 0.08);
  border-radius: 0.33rem;
  margin: 0 0.83rem 0.83rem 0;
  cursor: pointer;
  overflow: hidden;
  &:hover {
    filter: brightness(0.9);
  }
  &:nth-child(4n) {
    margin-right: 0;
  }
  .course-content {
    .content-title {
      margin-bottom: 0.25rem;
      .title {
        font-weight: bold;
        color: #333333;
        margin-right: 0.25rem;
      }
      .tag {
        min-width: 2.25rem;
        box-sizing: border-box;
        font-size: 12px;
        background: #f99a0b;
        border-radius: 0.1rem;
        text-align: center;
        color: #ffffff;
        padding: 0 0.25rem;
        transform: scale(0.8);
      }
    }
    .content-price {
      font-weight: 500;
      color: #ff3300;
      .num {
        font-size: 15px;
        line-height: 35rpx;
        margin-right: 8rpx;
      }
      .label {
        font-size: 12px;
        line-height: 28rpx;
        font-weight: bold;
      }
    }
  }
  .course-footer {
    position: absolute;
    width: 100%;
    bottom: 0.5rem;
    left: 0;
    box-sizing: border-box;
    padding: 0 0.5rem;
    color: #aaa;
  }
`;
