import React from 'react';
import { VerticalAlignTopOutlined } from '@ant-design/icons';
import dayjs from 'dayjs';
import styled from 'styled-components';
import { InterDataType, InterListType } from '@/api/interface';
import { AppPublishListType } from '@/api/interface/home';
import { GetIndustryListPagesType } from '@/api/interface/service';
import { formatLocationStr } from '@/utils/formatLocation';
import { bigNumberTransform } from '@/utils/money';

// 列表类型
type DetailType = InterDataType<AppPublishListType>[0];
// 行业类型
type IndustryType = InterListType<GetIndustryListPagesType>;

// 服务标签类型
const serviceTagList = [
  {
    label: '加急单',
    code: 'RUSH_ORDER',
    color: '#FF0000',
    text: '急',
  },
  {
    label: '置顶单',
    icon: <VerticalAlignTopOutlined />,
    code: 'TOP_ORDER',
    color: '#FF7A2C',
  },
];
// 保险列表
const insuranceList = [
  { name: '飞手险', value: 1 },
  { name: '机身险', value: 2 },
  { name: '三者险', value: 3 },
];
// 订单状态展示
const orderStateList = [
  { name: '抢单中', color: '#FE2910', value: ['100', '170'] },
  {
    name: '进行中',
    color: '#4D9BFF',
    value: ['150', '200', '300', '400', '500', '550', '600'],
  },
  { name: '已取消', color: '#999999', value: ['700'] },
  { name: '已完成', color: '#36C1AF', value: ['600'] },
];

const ServiceTaskItem: React.FC<{
  detail: DetailType;
  industryList: IndustryType;
  width?: string;
}> = ({ detail, industryList, width }) => {
  // 根据服务id获取服务详情
  const getIndustryItem = industryList.find((i) => i.id === detail?.serviceId);
  // 获取任务时间
  const getTaskRange = `${dayjs(detail?.taskStartTime).format('MM.DD')}~${dayjs(
    detail?.taskEndTime,
  ).format('MM.DD')}`;
  // 获取当前服务的标签类型
  const getServiceTag = () => {
    const item = serviceTagList?.find((i) => i.code === detail?.orderLevelEnum);
    return item || serviceTagList?.[0];
  };
  // 获取订单金额
  const getOrderAmount = () => {
    const money = (
      bigNumberTransform(detail?.orderAmount, true) as string
    )?.replace('.00', '');
    // formatMoney(props.detail?.orderAmount)
    return detail?.orderAmount > 1 ? money : detail?.orderAmount;
  };
  // 获取任务城市
  const getTaskCity = () => {
    const item = formatLocationStr(detail?.taskAddress);
    return item?.city || '深圳市';
  };
  // 获取保险列表
  const getInsuranceList =
    detail?.insurance
      ?.split(',')
      ?.map((i) => insuranceList.find((n) => n.value === Number(i))?.name)
      ?.join(' | ') || '无';
  // 根据订单状态回显对应的状态
  const getOrderStatus = () => {
    // 如果剩余接单人数为0，则直接返回已完成
    if (detail?.repertory === 0) return orderStateList.at(-1);
    // 根据订单状态返回对应的状态
    return orderStateList.find((i) => i.value.includes(detail?.orderStatus));
  };
  // 跳转详情
  const handleDetail = () => {
    console.log('handleDetail');
  };
  return (
    <ServiceTaskItemWrap
      onClick={handleDetail}
      className="select-none"
      style={width ? { width } : undefined}
    >
      {/* 头部区域 */}
      <div className="item-head">
        {getIndustryItem?.typeImg && (
          <img
            className="head-img"
            src={`${getIndustryItem?.typeImg}?x-oss-process=image/quality,q_25`}
            alt="服务类型图标"
          />
        )}
        <div className="head-name text-ellipsis" title={detail.serviceName}>
          {detail.serviceName || getIndustryItem?.typeName}
        </div>
        <div className="head-text">{getTaskRange}</div>
        <div className="head-num">
          <span>任务{detail.requireNum || 1}人</span>
          {!!detail.repertory && detail.requireNum > 1 && (
            <span>（剩余{detail.repertory}人）</span>
          )}
        </div>
      </div>

      {/* 标题区域 */}
      <div className="item-title">
        {getServiceTag() && detail?.orderLevelEnum !== 'REGULAR_ORDER' && (
          <div
            className="title-tag"
            style={{ background: getServiceTag().color }}
          >
            {getServiceTag()?.icon && getServiceTag()?.icon}
            {getServiceTag()?.text && (
              <div className="text">{getServiceTag()?.text}</div>
            )}
          </div>
        )}
        <div
          className="title-text text-ellipsis"
          title={detail.requireDescription}
        >
          {detail.requireDescription}
        </div>
        {detail.orderAmount && (
          <div className="title-price">
            <span className="label">￥</span>
            <span className="num">{getOrderAmount()}</span>
          </div>
        )}
      </div>

      {/* 内容区域 */}
      <div className="item-content">
        <div className="content-label">{getTaskCity()}</div>
        <div className="content-text">
          {dayjs(detail.createTime).format('MM-DD HH:mm')} 发布
        </div>
        {/* <div className="content-text"> */}
        {/*  <span>剩余：</span> */}
        {/*  <Statistic.Countdown */}
        {/*    value={dayjs(`${detail?.taskEndTime} 23:59:59`).valueOf()} */}
        {/*    format="D 天 H 时 m 分 s 秒" */}
        {/*    valueStyle={{ fontSize: '12px', color: '#999999' }} */}
        {/*  /> */}
        {/* </div> */}
      </div>
      {/* <div className="item-content dot"> */}
      {/*  <div className="content-label">保险要求</div> */}
      {/*  <div className="content-text">{getInsuranceList}</div> */}
      {/* </div> */}

      {/* 发布时间 */}
      {/* <div className="item-time"> */}
      {/*  {dayjs(detail.createTime).format('MM-DD HH:mm')} 发布 */}
      {/* </div> */}

      {/* 订单状态 */}
      {getOrderStatus() && (
        <div className="item-state" style={{ color: getOrderStatus()?.color }}>
          {getOrderStatus()?.name}
        </div>
      )}
    </ServiceTaskItemWrap>
  );
};

export default ServiceTaskItem;
// 样式
const ServiceTaskItemWrap = styled.div`
  position: relative;
  box-sizing: border-box;
  width: calc((100% - (0.83rem * 3)) / 4);
  height: 6.25rem;
  margin: 0 0.83rem 0.83rem 0;
  background: #ffffff;
  box-shadow: 0 0.17rem 0.67rem 0 rgba(24, 90, 190, 0.09);
  border-radius: 0.33rem;
  overflow: hidden;
  &:nth-child(4n) {
    margin-right: 0;
  }
  &:hover {
    cursor: pointer;
    background: #f6f6f6;
  }
  .item-head {
    position: relative;
    width: 100%;
    height: 1.5rem;
    display: flex;
    align-items: center;
    justify-content: flex-start;
    background: linear-gradient(
      91deg,
      #eaf6ff 0%,
      rgba(255, 255, 255, 0.04) 100%
    );
    box-sizing: border-box;
    padding: 0 0.5rem;
    margin-bottom: 0.33rem;
    .head-img {
      width: 0.83rem;
      height: 0.83rem;
      margin-right: 0.5rem;
    }
    .head-name {
      max-width: 10rem;
      font-weight: bold;
      margin-right: 0.5rem;
    }
    .head-text {
      color: #666;
    }
    .head-num {
      position: absolute;
      right: 0.5rem;
    }
  }
  .item-title {
    position: relative;
    display: flex;
    align-items: center;
    justify-content: flex-start;
    box-sizing: border-box;
    padding: 0 0 0 0.5rem;
    margin-bottom: 0.86rem;
    //margin-bottom: 0.25rem;
    .title-tag {
      width: 1rem;
      height: 1rem;
      border-radius: 0.25rem;
      display: flex;
      align-items: center;
      justify-content: center;
      line-height: 0.75rem;
      margin-right: 0.25rem;
      .text {
        font-weight: 400;
        color: #ffffff;
      }
      .icon {
        width: 0.75rem;
        height: 0.75rem;
      }
    }
    .title-text {
      font-weight: bold;
      max-width: 12rem;
    }
    .title-price {
      position: absolute;
      right: 0.5rem;
      width: 5rem;
      text-align: right;
      overflow: hidden;
      text-overflow: ellipsis;
      white-space: nowrap;
      word-break: break-all;
      transform: translateY(0) scaleX(0.9);
      .num {
        font-size: 1.5rem;
        color: #fe2910;
        font-weight: bold;
        //text-shadow: 1rpx 0 0 #fe2910, 0 -1rpx 0 #fe2910, -1rpx 0 0 #fe2910, 0 1rpx 0 #fe2910;
        //text-shadow: 0 0 0 #fe2910, 0 0 0 #fe2910;
      }
      .label {
        color: #fe2910;
        font-size: 1rem;
        font-weight: bold;
      }
    }
  }
  .item-content {
    position: relative;
    width: 100%;
    box-sizing: border-box;
    padding: 0 0.5rem 0 1rem;
    display: flex;
    align-items: center;
    justify-content: flex-start;
    &::after {
      position: absolute;
      top: calc((100% - 4px) / 2);
      left: 6px;
      content: '';
      width: 4px;
      height: 4px;
      background: #fe2910;
      border-radius: 50%;
    }
    .content-label {
      color: #666666;
      font-weight: bold;
      margin-right: 0.5rem;
    }
    .content-text {
      display: flex;
      align-items: center;
      justify-content: flex-start;
      color: #999999;
    }
  }
  .dot::after {
    background: #36c1af !important;
  }
  .item-time {
    position: absolute;
    right: 0.5rem;
    bottom: 0.25rem;
  }
  .item-state {
    position: absolute;
    right: 0.5rem;
    bottom: 0.86rem;
  }
`;
