import React, { useEffect, useState } from 'react';
import {
  EnvironmentFilled,
  LogoutOutlined,
  ReloadOutlined,
} from '@ant-design/icons';
import { App, Button, Dropdown, Input, MenuProps, Modal } from 'antd';
import dayjs from 'dayjs';
import Cookies from 'js-cookie';
import { throttle } from 'lodash';
import { useRouter } from 'next/router';
import { useDispatch, useSelector } from 'react-redux';
import { CommonAPI } from '@/api';
import { HeaderWrap } from '@/components/layout/header/styled';
import QrcodePopover from '@/components/qrcodePopover';
import { RootState } from '@/store';
import { AddressState, setAddress } from '@/store/module/address';
import { setGlobalData } from '@/store/module/globalData';
import { setSystem, SystemState } from '@/store/module/system';
import { setUserInfo, UserInfoState } from '@/store/module/userInfo';
import getLocationByIP from '@/utils/getLocationByIP';

const HeaderView: React.FC<{
  placeholder: boolean;
  autoChange: boolean;
  topDistance: number;
}> = ({ placeholder, autoChange, topDistance }) => {
  // 静态方法
  const { message } = App.useApp();
  // token
  const token = Cookies.get('SHAREFLY-WEB-TOKEN');
  // 当前的路由数据
  const router = useRouter();
  // store
  const dispatch = useDispatch();
  // system
  const system = useSelector((state: RootState) => state.system) as SystemState;
  // address
  const address = useSelector(
    (state: RootState) => state.address,
  ) as AddressState;
  // userInfo
  const userInfo = useSelector(
    (state: RootState) => state.userInfo,
  ) as UserInfoState;
  // 打开登录弹窗
  const handleLogin = () => {
    dispatch(
      setGlobalData({
        loginModalVisible: true,
      }),
    );
  };
  // 获取用户信息
  const getAccountInfo = async () => {
    if (!system?.token || !token) return;
    const res = await CommonAPI.getAccountInfo();
    if (res && res.code === '200') {
      dispatch(setUserInfo(res.result));
      if (!res.result) message.error('获取用户信息失败');
    }
  };
  // 计算天数与当前时间的差值
  const getDiffDay = (date: string) => dayjs().diff(dayjs(date), 'day');
  // 重新获取用户的地址信息
  const handleReload = () => {
    getLocationByIP().then((res) => {
      dispatch(setAddress(res));
    });
  };
  // 跳转详情
  const handleDetail = async (item: { label: string; value: string }) => {
    await router.push(item.value);
  };
  // 是否自动变化
  const [isAutoChange, setIsAutoChange] = useState(false);
  // 定义一个副作用函数，用于获取滚动距离并更新状态
  const handleScroll = () => {
    // 获取滚动距离，兼容多种浏览器
    const scrollTop = Math.max(
      window.scrollY,
      document.documentElement.scrollTop,
      document.body.scrollTop,
    );
    // 更新状态
    setIsAutoChange(scrollTop >= topDistance);
  };
  // 节流函数
  const handleThrottle = throttle(handleScroll, 500);
  // 组件挂载
  useEffect(() => {
    if (!address?.city) {
      // 获取当前地址
      getLocationByIP().then((res) => {
        dispatch(setAddress(res));
      });
    }
    // console.log('当前是否登录 --->', system, token);
    // 当前是否登录
    if (!token) {
      dispatch(setSystem({ token: undefined }));
      dispatch(setUserInfo(null));
    } else {
      if (!system?.token) dispatch(setSystem({ token }));
      getAccountInfo().then();
    }
    if (autoChange) window.addEventListener('scroll', handleThrottle);
    // 组件卸载前，移除监听器
    return () => {
      if (autoChange) {
        window.removeEventListener('scroll', handleThrottle);
        handleThrottle.cancel();
      }
    };
  }, []);
  // 顶部Tab列表
  const tabList: { label: string; value: string; isQrcode?: boolean }[] = [
    { label: '个人中心', value: '/mine', isQrcode: false },
    { label: '购物车', value: 'page-cart/cart-list/index', isQrcode: true },
    { label: '我的订单', value: '/order', isQrcode: false },
    { label: '消息', value: 'pages/message/index', isQrcode: true },
    { label: '联系客服', value: 'page-mine/help-center/index', isQrcode: true },
  ];
  // 右上角按钮
  const items: MenuProps['items'] = [
    {
      key: '1',
      label: (
        <div style={{ textAlign: 'left', marginBottom: '20px' }}>
          <div style={{ fontWeight: 'bold', marginBottom: '4px' }}>
            用户信息：
          </div>
          <div>昵称：{userInfo?.nickName}</div>
          <div>手机号：{userInfo?.phoneNum}</div>
          <div>来到云享飞 {getDiffDay(userInfo?.createTime)} 天</div>
        </div>
      ),
    },
    {
      key: '0',
      label: (
        <Button
          style={{ color: '#666666' }}
          type="link"
          icon={<LogoutOutlined />}
          onClick={() => {
            Modal.confirm({
              title: '退出登录',
              content: '退出后未保存数据将会丢失，确定登出吗？',
              onOk: () => {
                dispatch(setUserInfo(null));
                dispatch(setSystem({ token: undefined }));
                Cookies.remove('SHAREFLY-WEB-TOKEN');
              },
            });
          }}
        >
          退出登录
        </Button>
      ),
    },
  ];
  return (
    <>
      <HeaderWrap
        style={{
          background:
            placeholder || isAutoChange ? '#2A2A2A' : 'rgba(86, 86, 86, 0.25)',
        }}
      >
        <div className="header-wrap">
          <div className="header-location">
            <div className="location-icon">
              <EnvironmentFilled
                style={{ color: '#FF552D', fontSize: '0.86rem' }}
              />
            </div>
            <Button type={'link'} className="location-address">
              {address?.city || '定位中...'}
            </Button>
            <Button
              type={'link'}
              icon={<ReloadOutlined style={{ fontSize: '10px' }} />}
              className="location-reload"
              title={'刷新位置'}
              onClick={() => handleReload()}
            ></Button>
            <div className="location-hello">Hi，欢迎来到云享飞</div>
            {router?.pathname !== '/' && (
              <Button
                type={'link'}
                className="text-[#fff]"
                title="返回首页"
                onClick={() => router.push('/')}
              >
                首页
              </Button>
            )}
          </div>
          <div className="header-search">
            <Input.Search
              className="search-box"
              placeholder="科比特航空"
              enterButton="搜索"
              size={'middle'}
              bordered={true}
            />
          </div>
          <div className="header-nav">
            <div className="nav-tab">
              {tabList?.map((i, j) =>
                !i.isQrcode ? (
                  <Button
                    type={'link'}
                    className="tab-item"
                    key={j}
                    onClick={() => handleDetail(i)}
                  >
                    {i.label}
                  </Button>
                ) : (
                  <QrcodePopover path={i.value} key={j}>
                    <Button type={'link'} className="tab-item">
                      {i.label}
                    </Button>
                  </QrcodePopover>
                ),
              )}
            </div>
            {!!userInfo?.id && (
              <div className="nav-user">
                <Dropdown
                  overlayStyle={{ textAlign: 'center' }}
                  menu={{ items }}
                  placement="bottomRight"
                  arrow
                >
                  <div
                    className="user-avatar"
                    style={{ backgroundImage: `url(${userInfo?.userImg})` }}
                  ></div>
                </Dropdown>
              </div>
            )}
            <div className="nav-action">
              {!userInfo?.id && (
                <Button
                  type={'primary'}
                  className="action-item"
                  onClick={handleLogin}
                >
                  登录
                </Button>
              )}
              <QrcodePopover path={'page-service/service-flyer/index'}>
                <Button type={'primary'} className="action-item">
                  发布任务
                </Button>
              </QrcodePopover>
              <QrcodePopover path={'page-identity/identity-list/index'}>
                <Button type={'primary'} className="action-item">
                  加盟入驻
                </Button>
              </QrcodePopover>
            </div>
          </div>
        </div>
      </HeaderWrap>
      {(placeholder || isAutoChange) && (
        <div
          className="header-wrap"
          style={{ width: '100%', height: '3rem' }}
        ></div>
      )}
    </>
  );
};

export default HeaderView;
