import React, { useState } from 'react';
import {
  DownOutlined,
  LikeOutlined,
  MessageOutlined,
  UpOutlined,
} from '@ant-design/icons';
import { App, Button, Input } from 'antd';
import styled from 'styled-components';
import { HomeAPI } from '@/api';
import { InterListType } from '@/api/interface';
import { AllCommentListType, ReplyListType } from '@/api/interface/home';

// 详情类型
type DetailType = InterListType<AllCommentListType>[0];
// 列表类型
type ListType = InterListType<ReplyListType>;

const ForumCommentItem: React.FC<{
  detail: DetailType;
  reply?: ListType;
  origin?: DetailType;
  onRefresh?: () => void;
}> = ({ detail, reply, origin, onRefresh }) => {
  // 静态组件
  const { message } = App.useApp();
  // 是否显示回复输入
  const [isInput, setIsInput] = useState(false);
  // 是否展开评论列表
  const [isReply, setIsReply] = useState(false);
  // 回复列表
  const [replyList, setReplyList] = useState<ListType>([]);
  // 获取回复列表
  const getReplyList = async () => {
    const res = await HomeAPI.getReplyList({
      dynamicId: detail?.dynamicId,
      id: detail?.id,
      pageNo: 1,
      pageSize: 999,
    });
    if (res && res.code === '200') {
      // console.log('回复列表 ===>', res.result?.list);
      setReplyList(res.result?.list || []);
    }
  };
  // 展开评论列表
  const handleReply = async (e: Boolean) => {
    setIsReply(!e);
    if (!e) {
      getReplyList().then();
    } else {
      setReplyList([]);
    }
  };
  // 获取@人的名字
  const getReplyName = () => {
    if (!reply) return undefined;
    const item = reply?.find((i) => i?.id === detail?.pid);
    return item ? (
      <>
        回复&nbsp;
        <Button type={'link'} className="px-0">
          @{item?.userAccountVO?.nickName}
        </Button>
        &nbsp; ：
      </>
    ) : undefined;
  };
  // 获取@人的名字
  const getReplyNameStr = () => {
    if (!reply) return `回复 @${detail?.userAccountVO?.nickName} ：`;
    const item = reply?.find((i) => i?.id === detail?.pid);
    return item
      ? `回复 @${item?.userAccountVO?.nickName} ：`
      : `回复 @${detail?.userAccountVO?.nickName} ：`;
  };
  // 回复数据
  const [replyText, setReplyText] = useState<string>();
  // 提交数据
  const handleSubmit = async () => {
    if (!replyText) {
      await message.warning('请输入内容');
      return;
    }
    // 提交数据
    const res = await HomeAPI.publishComment({
      content: replyText,
      dynamicId: detail?.dynamicId,
      pid: detail?.id,
      reviewId: origin?.id || detail?.id,
    });
    if (res && res.code === '200') {
      message.success('回复成功');
      setReplyText(undefined);
      setIsInput(false);
      onRefresh?.();
    }
  };
  // 点赞
  const handleReviewLikes = async () => {
    const res = await HomeAPI.reviewLikes({
      id: detail?.id,
      status: !detail?.status,
    });
    if (res && res.code === '200') {
      message.success(!detail?.status ? '点赞成功' : '取消点赞');
      onRefresh?.();
    }
  };
  return (
    <ForumCommentWrap>
      <img
        className={'image'}
        src={detail?.userAccountVO?.userImg}
        alt={detail?.userAccountVO?.userName}
      />
      <div className="content">
        <div className="name">
          {detail?.userAccountVO?.nickName ||
            `云享飞用户_${detail?.userAccountId}`}
        </div>
        <div className="text">
          {getReplyName()}
          {detail?.content}
        </div>
        <div className="action">
          <div className="date">{detail?.createTime}</div>
          <div className="button">
            <Button
              type={'text'}
              icon={<MessageOutlined />}
              onClick={() => setIsInput(!isInput)}
            >
              {isInput ? '取消回复' : '回复'}
            </Button>
            <Button
              type={'text'}
              icon={<LikeOutlined />}
              className={!detail?.status ? 'text-333' : 'text-primary'}
              onClick={handleReviewLikes}
            >
              &nbsp;{detail?.likeCount || 0}
            </Button>
          </div>
        </div>
        {isInput && (
          <div className="textarea">
            <Input.TextArea
              placeholder={getReplyNameStr()}
              showCount
              style={{ height: 86 }}
              maxLength={140}
              onChange={(e) => setReplyText(e.target.value)}
            />
            <div className="action">
              <Button type="primary" shape="round" onClick={handleSubmit}>
                发布
              </Button>
            </div>
          </div>
        )}
        {!!replyList?.length &&
          replyList?.map((n, m) => (
            <ForumCommentItem
              detail={n as any}
              key={m}
              reply={replyList}
              origin={detail}
              onRefresh={getReplyList}
            />
          ))}
        {!!detail?.replyCount && (
          <div className="flex">
            <div className="reply" onClick={() => handleReply(isReply)}>
              {!isReply ? `展开${detail?.replyCount}条回复` : `收起`}
            </div>
            {!isReply ? (
              <DownOutlined className="ml-1 text-xs text-999" />
            ) : (
              <UpOutlined className="ml-1 text-xs text-999" />
            )}
          </div>
        )}
      </div>
    </ForumCommentWrap>
  );
};

export default ForumCommentItem;
// 样式
const ForumCommentWrap = styled.div`
  position: relative;
  width: 100%;
  box-sizing: border-box;
  display: flex;
  align-items: flex-start;
  justify-content: flex-start;
  &:not(:first-child) {
    margin-top: 1rem;
  }
  .image {
    width: 2.25rem;
    height: 2.25rem;
    border-radius: 50%;
    margin-right: 1rem;
  }
  .content {
    position: relative;
    width: calc(100% - 3.5rem);
    box-sizing: border-box;
    .name {
      font-weight: bold;
    }
    .text {
      width: 100%;
      color: #666666;
      margin-bottom: 0.5rem;
    }
    .action {
      position: relative;
      width: 100%;
      display: flex;
      justify-content: space-between;
      align-items: center;
      .date {
        color: #999999;
      }
    }
    .textarea {
      position: relative;
      width: 100%;
      box-sizing: border-box;
      margin-top: 0.5rem;
      .action {
        margin-top: 1.5rem;
        width: 100%;
        display: flex;
        justify-content: flex-end;
      }
    }
    .reply {
      cursor: pointer;
      &:hover {
        color: #ff392b;
      }
    }
  }
`;
