import React, { useEffect, useState } from 'react';
import { Image, Table } from 'antd';
import { ColumnsType } from 'antd/es/table';
import styled from 'styled-components';
import { InterDataType } from '@/api/interface';
import { UavOrderDetailType } from '@/api/interface/order';
import { OrderAPI } from '@/api/modules/order';

// 接口返回的类型
type DataType = InterDataType<UavOrderDetailType>;
// 参数类型
type PropsType = {
  detail: DataType | undefined;
};

const MallOrderSpecView: React.FC<PropsType> = (props) => {
  const { detail } = props;
  // 表格结构
  const columns: ColumnsType<DataType['skuDTOList'][0]> = [
    {
      title: '商品',
      align: 'center',
      width: '20%',
      render: (_text, record) => (
        <div className="mall-sku-item">
          <Image src={record.skuImage} className="sku-img" />
          <div className="sku-info">
            <div className="info-name">{record.tradeName}</div>
            <div className="info-spec">
              {Object.entries(JSON.parse(record.productSpec)).map(
                (v, index) => (
                  <span key={index}>{v.join('：')}</span>
                ),
              )}
            </div>
          </div>
        </div>
      ),
    },
    {
      title: '单价（元）',
      dataIndex: 'unitPrice',
      align: 'center',
      render: (text) => `￥${text?.toLocaleString()}`,
    },
    {
      title: '数量',
      dataIndex: 'orderNum',
      align: 'center',
    },
    {
      title: '订单状态',
      align: 'center',
      render: () =>
        orderStatusList.find((v) => v.value === detail?.statusCode)?.label,
    },
    {
      title: '订单金额（元）',
      align: 'center',
      render: () => `￥${detail?.orderTotalAmount}`,
    },
  ];
  // 订单状态
  const [orderStatusList, setOrderStatusList] = useState<
    { value: number; label: string }[]
  >([]);

  // 获取订单状态
  const getOrderStatus = () => {
    OrderAPI.uavOrderStatusList().then(({ result }) => {
      if (result) {
        const list = result.map((item) => ({
          value: item.code,
          label: item.status,
        }));
        setOrderStatusList(list);
      }
    });
  };

  useEffect(() => {
    getOrderStatus();
  }, []);
  return (
    <DetailInfoWrap>
      <div className="detail-title">订单明细</div>
      <Table
        size="small"
        dataSource={detail?.skuDTOList || []}
        columns={columns}
        rowKey="id"
        bordered
        pagination={false}
        // rowSelection={{ selectedRowKeys, onChange: onSelectChange }}
      />
    </DetailInfoWrap>
  );
};

export default MallOrderSpecView;
// 样式
const DetailInfoWrap = styled.div`
  .detail-title {
    font-weight: bold;
    margin: 1rem 0 0.5rem 0;
  }
  .mall-sku-item {
    display: flex;
    align-items: center;
    margin-bottom: 10px;
    .sku-img {
      width: 48px;
      height: 48px;
    }
    .sku-info {
      margin-left: 10px;
      line-height: 16px;
      text-align: left;
      .info-name {
        color: #1677ff;
      }
      .info-spec {
        margin-top: 10px;
        span:not(:last-child) {
          &:after {
            content: '；';
          }
        }
      }
    }
  }
`;
