import React from 'react';
import { useDispatch, useSelector } from 'react-redux';
import styled from 'styled-components';
import { InterDataType } from '@/api/interface';
import { GetCompanyInspectionById } from '@/api/interface/service';
import ProductSwiperView from '@/components/product-swiper';
import { GlobalDataState, setGlobalData } from '@/store/module/globalData';
import { UserInfoState } from '@/store/module/userInfo';
import { formatMoney } from '@/utils/money';

// 商品详情类型
type DetailType = InterDataType<GetCompanyInspectionById>;

const ServiceHeadView: React.FC<{ detail: DetailType }> = ({ detail }) => {
  // 当前的路由数据
  // const router = useRouter();
  // store
  const dispatch = useDispatch();
  // userInfo
  const userInfo = useSelector((state: any) => state.userInfo) as UserInfoState;
  // globalData
  const globalData = useSelector(
    (state: any) => state.globalData,
  ) as GlobalDataState;
  // 提交事件
  const handleSubmit = async (type: number) => {
    // 判断是否登录
    if (!userInfo?.id) {
      dispatch(
        setGlobalData({
          loginModalVisible: true,
        }),
      );
      return;
    }
    // 立即购买
    if (type === 1) {
      dispatch(
        setGlobalData({
          qrcodeModalVisible: true,
          qrcodeModalPath: 'page-service/service-flyer/index',
          qrcodeModalScene: `id=${Number(detail?.id)}`,
        }),
      );
      return;
    }
    // 在线沟通
    if (type === 2) {
      dispatch(
        setGlobalData({
          qrcodeModalVisible: true,
          qrcodeModalPath: 'page-service/service-detail/index',
          qrcodeModalScene: `id=${Number(detail?.id)}`,
        }),
      );
    }
  };
  // 转换swiper图片
  const getSwiperList = () => {
    return detail?.inspectionFileDTOS
      ?.filter((i) => i.fileType !== 1)
      ?.sort((a, b) => b.first - a.first)
      ?.map((i) => ({ id: i.id, url: i.fileUrl }));
  };
  // 获取商品的单位
  const getPriceUnit = () => {
    const unit = globalData?.priceUnitList?.find(
      (i) => i.id === detail?.inspectionPriceUnitId,
    );
    return unit?.unitName || '次';
  };
  return (
    <ServiceHeadWrap>
      <div className="product-swiper">
        <ProductSwiperView list={getSwiperList()} />
      </div>
      <div className="product-content">
        <div className="content-title">
          【{detail?.industryTypeDTO?.typeName}】
          {detail?.inspectionDTO?.inspectionName}
        </div>
        <div className="content-price flex-start">
          <div className="price-label">价格</div>
          <div className="price-money">
            {detail?.price ? (
              <>
                <span className="label">￥</span>
                <span
                  className="num"
                  title={`${formatMoney(detail?.price)}元起每${getPriceUnit()}`}
                >
                  {formatMoney(detail?.price)}
                </span>
                <span
                  className="unit text-ellipsis"
                  title={`${formatMoney(detail?.price)}元起每${getPriceUnit()}`}
                >
                  起/{getPriceUnit()}
                </span>
              </>
            ) : (
              <span className="label">咨询报价</span>
            )}
          </div>
        </div>
        <div className="content-item flex-start">
          <div className="item-label">团队</div>
          <div className="item-content flex-start">
            <img
              className="team-label"
              src="https://file.iuav.com/file/sharefly-service-label01.png"
              alt="专业飞手"
            />
            <div className="team-text">飞手已通过认证培训</div>
          </div>
        </div>
        <div className="content-item flex-start item-bottom">
          <div className="item-label">服务</div>
          <div className="item-content flex-start">
            <div className="item-tag">7x24小时服务</div>
            <div className="item-tag">已售{Math.floor(detail.id * 2.22)}</div>
          </div>
        </div>
        <div className="content-action flex-start select-none">
          <div className="action-item" onClick={() => handleSubmit(2)}>
            在线沟通
          </div>
          <div className="action-item" onClick={() => handleSubmit(1)}>
            马上预约
          </div>
        </div>
      </div>
    </ServiceHeadWrap>
  );
};

export default ServiceHeadView;

// 样式
const ServiceHeadWrap = styled.div`
  position: relative;
  width: calc((100% - 0.83rem) / 10 * 7.5);
  min-height: 28rem;
  background: #ffffff;
  border: 0.04rem solid #e3e3e3;
  box-sizing: border-box;
  margin-right: 0.83rem;
  display: flex;
  align-items: flex-start;
  justify-content: flex-start;
  padding: 2rem 1rem 2rem 1rem;
  .product-swiper {
    position: relative;
    width: 22rem;
    height: 26rem;
    box-sizing: border-box;
  }
  .product-content {
    position: relative;
    width: calc(100% - 22rem);
    height: 100%;
    box-sizing: border-box;
    padding: 0 0 0 1rem;
    //background: lightblue;
    .content-title {
      width: 100%;
      font-size: 24px;
      font-weight: 500;
      color: #212121;
      margin-bottom: 0.71rem;
      // 双行省略
      overflow: hidden;
      text-overflow: ellipsis;
      display: -webkit-box;
      -webkit-line-clamp: 2;
    }
    .content-desc {
      font-weight: 400;
      color: #666666;
      margin-bottom: 0.71rem;
    }
    .content-price {
      position: relative;
      width: 100%;
      background: #f3f3f3;
      box-sizing: border-box;
      padding: 0.8rem;
      margin-bottom: 1rem;
      align-items: baseline;
      .price-label {
        width: 2.5rem;
        color: #999999;
        text-align: justify;
        text-align-last: justify;
        margin-right: 1rem;
        transform: translateY(-0.1rem);
      }
      .price-money {
        font-size: 24px;
        //line-height: 1;
        color: #ff6700;
        .label {
          font-size: 18px;
        }
        .unit {
          font-size: 13px;
          color: #999999;
          margin-left: 0.5rem;
        }
      }
    }
    .content-item {
      margin-bottom: 1rem;
      padding: 0 0.8rem;
      //flex-wrap: nowrap;
      .item-label {
        width: 2.5rem;
        color: #999999;
        text-align: justify;
        text-align-last: justify;
        margin-right: 1rem;
      }
      .item-content {
        width: calc(100% - 3.5rem);
        .team-label {
          height: 2rem;
          width: 6rem;
          object-fit: contain;
          margin-right: 0.5rem;
        }
        .team-text {
          color: #666666;
        }
        .item-tag {
          box-sizing: border-box;
          background: #f4f4f4;
          border-radius: 0.13rem;
          padding: 0 0.33rem 0 0.33rem;
          color: #333333;
          &:not(:last-child) {
            margin-right: 0.5rem;
          }
        }
        .content-address {
          font-size: 12px;
          &:first-child {
            margin-bottom: 0.3rem;
            .text {
              color: #8e8e8e;
            }
          }
          .text {
            margin-left: 0.3rem;
            font-weight: 400;
          }
        }
        .content-spec {
          //min-width: max-content;
          height: 2rem;
          background: #f2f2f2;
          border-radius: 0.08rem;
          text-align: center;
          line-height: 2rem;
          font-weight: 400;
          box-sizing: border-box;
          padding: 0 1rem;
          border: 0.04rem solid #f2f2f2;
          margin: 0 0.5rem 0.5rem 0;
          &:last-child {
            margin-right: 0;
          }
        }
        .spec-active {
          background: #ffede8;
          border: 0.04rem solid #ff552d;
          color: #ff552d;
        }
      }
    }
    .item-bottom {
      margin-bottom: 6.8rem;
    }
    .content-action {
      //position: absolute;
      //left: 1rem;
      //bottom: 0;
      position: relative;
      width: 100%;
      margin-top: 2rem;
      .action-item {
        width: 8.63rem;
        height: 2.33rem;
        background: #fff0e5;
        border: 0.04rem solid #ff552d;
        text-align: center;
        line-height: 2.33rem;
        font-size: 13px;
        font-weight: 400;
        color: #ff552d;
        cursor: pointer;
        &:hover,
        &:active {
          filter: brightness(0.95);
        }
      }
      .action-item:not(:last-child) {
        margin-right: 0.8rem;
      }
      .action-item:last-child {
        background: #ff552d;
        color: #fff;
      }
    }
  }
`;
