import React, { useEffect, useState } from 'react';
import { useRouter } from 'next/router';
import styled from 'styled-components';
import { HomeAPI } from '@/api';
import { InterListType } from '@/api/interface';
import { IndustryListPagesType } from '@/api/interface/home';

// 列表类型
type ListType = InterListType<IndustryListPagesType>;

const HomeServiceView = () => {
  // 路由钩子
  const router = useRouter();
  // 刷新子组件
  const [refresh, setRefresh] = useState<boolean>(true);
  // 服务列表
  const [industryList, setIndustryList] = useState<ListType>([]);
  // 当前服务列表索引
  const [currentIndex, setCurrentIndex] = useState<number>(0);
  // 获取分类列表
  const getIndustryListPages = async () => {
    const res = await HomeAPI.IndustryListPages({
      pageNo: 1,
      pageSize: 999,
    });
    if (res && res.code === '200') {
      setIndustryList(res.result?.list || []);
      // console.log('获取分类列表 --->', res.result?.list || []);
    }
  };
  // 列表筛选事件
  const handleSelect = (index: number) => {
    setRefresh(false);
    setCurrentIndex(index);
    setTimeout(() => {
      setRefresh(true);
    }, 0);
  };
  // 跳转详情
  const handleDetail = (item: ListType[0]['inspectionDTOS'][0]) => {
    router.push(`/service/${item?.industryTypeId}/${item?.id}`).then();
  };
  // componentDidMount
  useEffect(() => {
    getIndustryListPages().then();
  }, []);
  return (
    <HomeServiceWrap>
      <div className="service-tab flex-start">
        {industryList?.map((i, j) => (
          <div
            className={`tab-item ${j === currentIndex && 'item-active'}`}
            key={j}
            onClick={() => handleSelect(j)}
          >
            {i?.typeName}
          </div>
        ))}
      </div>
      <div className="service-list flex-start scroll-view">
        {refresh &&
          industryList?.[currentIndex]?.inspectionDTOS?.map((i, j) => (
            <div
              className="service-item animate__animated animate__fast animate__fadeIn overflow-hidden rounded"
              key={j}
              onClick={() => handleDetail(i)}
            >
              <img
                className="item-image"
                src={`${i?.inspectionImg}?x-oss-process=image/quality,q_5`}
                alt={i.inspectionName}
              />
              <div className="item-shadow">{i.inspectionName}</div>
            </div>
          ))}
      </div>
    </HomeServiceWrap>
  );
};

const HomeServiceWrap = styled.div`
  position: relative;
  width: 100%;
  min-height: 16.75rem;
  box-sizing: border-box;
  padding-bottom: 2rem;
  .service-tab {
    position: relative;
    width: 100%;
    height: 3.38rem;
    flex-wrap: wrap;
    .tab-item {
      font-size: 13px;
      font-weight: 500;
      color: #333333;
      margin-right: 0.83rem;
      cursor: pointer;
      &:active,
      &:hover {
        color: #ff0000;
      }
    }
    .item-active {
      color: #ff0000;
      box-sizing: border-box;
      padding: 0.25rem 0.5rem;
      border-radius: 0.25rem;
      border: 0.04rem solid #ff552d;
      text-align: center;
    }
  }
  .service-list {
    position: relative;
    width: 100%;
    height: 22rem;
    flex-direction: column;
    align-items: flex-start;
    align-content: flex-start;
    overflow: hidden;
    overflow-x: scroll;
    .service-item {
      position: relative;
      width: calc((100% - 0.83rem * 2) / 4);
      height: 10rem;
      //background: lightblue;
      .item-shadow {
        position: absolute;
        width: 100%;
        height: 2.88rem;
        bottom: 0;
        left: 0;
        background: linear-gradient(
          180deg,
          rgba(0, 0, 0, 0) 0%,
          rgba(0, 0, 0, 0.68) 100%
        );
        text-align: center;
        font-weight: 500;
        color: #ffffff;
        font-size: 13px;
        line-height: 2.88rem;
      }
      .item-image {
        width: 100%;
        height: 100%;
        object-fit: cover;
      }
      &:first-child {
        height: calc(100% - 0.83rem);
        margin-right: 0.83rem;
      }
      &:nth-child(2n) {
        margin-bottom: 0.83rem;
        margin-right: 0.83rem;
      }
      &:hover,
      &:active {
        filter: brightness(0.9);
      }
    }
  }
`;
export default HomeServiceView;
