import React, { useEffect, useState } from 'react';
import { Empty } from 'antd';
import styled from 'styled-components';
import { HomeAPI } from '@/api';
import { InterListType, InterReqType } from '@/api/interface';
import { LeaseGoodsListType, ListBrandInfoType } from '@/api/interface/home';
import BreadcrumbView from '@/components/breadcrumb';
import CategorySelectView, { CategoryType } from '@/components/categorySelect';
import LayoutView from '@/components/layout';
import ProductListView from '@/components/productList';
import RentListItem from '@/components/rentListItem';

// 分类列表类型
type CategoryListType = InterListType<ListBrandInfoType>;
// 列表类型
type ListType = InterListType<LeaseGoodsListType>;
// 请求参数类型
type ReqType = InterReqType<LeaseGoodsListType>;

const RentView: React.FC<{
  categoryList: CategoryListType;
}> = (props) => {
  // 分类列表
  const [categoryList, setCategoryList] = useState<CategoryType>([]);
  // 转换分类列表
  const getCategoryList = () => {
    setCategoryList(
      props?.categoryList?.map((i) => ({
        value: i.id,
        label: i.brandName,
        children: i.modeInfoList?.map((n) => ({
          value: n.id,
          label: n.modeName,
        })),
      })),
    );
  };
  // 分页数据
  const [pagination, setPagination] = useState({
    pageNo: 1,
    pageSize: 18,
    totalCount: 0,
  });
  // 列表数据
  const [dataList, setDataList] = useState<ListType>([]);
  // 获取商品列表
  const getLeaseGoodsList = async (data: ReqType) => {
    const res = await HomeAPI.leaseGoodsList({
      pageNo: pagination.pageNo,
      pageSize: pagination.pageSize,
      ...data,
    });
    if (res && res.code === '200') {
      console.log('获取商品列表 --->', res);
      const { list, totalCount, pageNo, pageSize } = res.result;
      setDataList(list || []);
      setPagination({
        ...pagination,
        totalCount,
        pageNo,
        pageSize,
      });
    }
  };
  // 分类筛选
  const handleSelect = async (e: { main?: number; second?: number[] }) => {
    if (e.second?.length && e.second?.length > 1) {
      await getLeaseGoodsList({
        brandInfoId: e?.main,
        pageNo: 1,
      });
    } else {
      await getLeaseGoodsList({
        modelInfoId: e?.second?.[0],
        pageNo: 1,
      });
    }
  };
  // 翻页回调
  const handlePageChange = async (pageNo: number, pageSize: number) => {
    await getLeaseGoodsList({ pageNo, pageSize });
  };
  // 组件挂载
  useEffect(() => {
    if (!props) return;
    getCategoryList();
  }, [props]);
  return (
    <LayoutView placeholder={true}>
      <RentViewWrap>
        {/* 面包屑 */}
        <BreadcrumbView />
        {/* 类型筛选 */}
        <CategorySelectView
          list={categoryList}
          isMultiple={false}
          onSelect={handleSelect}
        />
        {/* 产品列表 */}
        <ProductListView pagination={pagination} onChange={handlePageChange}>
          {dataList?.length ? (
            dataList?.map((i, j) => <RentListItem key={j} detail={i} />)
          ) : (
            <div className="list-empty flex-center">
              <Empty />
            </div>
          )}
        </ProductListView>
      </RentViewWrap>
    </LayoutView>
  );
};

export default RentView;
// 样式
const RentViewWrap = styled.div`
  position: relative;
  max-width: 1190px;
  box-sizing: border-box;
  padding: 2rem 0 0 0;
  margin: 0 auto;
`;
