import React from 'react';
import { PropertySafetyFilled, ShoppingCartOutlined } from '@ant-design/icons';
import { useRouter } from 'next/router';
import { useSelector } from 'react-redux';
import styled from 'styled-components';
import { InterListType } from '@/api/interface';
import { GetListAPPCompanyInspectionPageType } from '@/api/interface/service';
import { GlobalDataState } from '@/store/module/globalData';
import { formatMoney } from '@/utils/money';

// 商品详情类型
type GoodsInfoListType = InterListType<GetListAPPCompanyInspectionPageType>[0];

const ServiceItemView: React.FC<{
  detail: GoodsInfoListType;
}> = ({ detail }) => {
  // 路由钩子
  const router = useRouter();
  // globalData
  const globalData = useSelector(
    (state: any) => state.globalData,
  ) as GlobalDataState;
  // 跳转商品详情
  const handleDetail = () => {
    router.push(`/service/detail/${detail?.id}`).then();
  };
  // 获取商品的单位
  const getPriceUnit = () => {
    const unit = globalData?.priceUnitList?.find(
      (i) => i.id === detail?.inspectionPriceUnitId,
    );
    return unit?.unitName || '次';
  };
  return (
    <ProductItemWrap onClick={handleDetail}>
      <div className="product-image">
        <img
          className="image"
          src={`${detail?.inspectionFirstImg}?x-oss-process=image/quality,q_10`}
          alt="图片"
        />
      </div>
      <div className="product-title">
        【{detail?.industryTypeDTO?.typeName}】
        {detail?.inspectionDTO?.inspectionName}
      </div>
      <div className="product-desc flex-between">
        <div className="money">
          {detail?.price ? (
            <>
              <span className="label">￥</span>
              <span
                className="num"
                title={`${formatMoney(detail?.price)}元起每${getPriceUnit()}`}
              >
                {formatMoney(detail?.price)}
              </span>
              <div
                className="unit text-ellipsis"
                title={`${formatMoney(detail?.price)}元起每${getPriceUnit()}`}
              >
                起/{getPriceUnit()}
              </div>
            </>
          ) : (
            <span className="label">咨询报价</span>
          )}
        </div>
        <div className="text text-ellipsis">
          成交{Math.floor(detail.id * 3.14)}件
        </div>
      </div>
      <div className="product-store flex-start">
        <PropertySafetyFilled style={{ color: '#FF552D' }} />
        <div className="title text-ellipsis" title={detail?.companyName}>
          {detail?.companyName}
        </div>
      </div>
      <div className="product-cart">
        <ShoppingCartOutlined style={{ color: '#ffffff', fontSize: '16px' }} />
      </div>
    </ProductItemWrap>
  );
};

export default ServiceItemView;
const ProductItemWrap = styled.div`
  position: relative;
  box-sizing: border-box;
  width: calc((100% - (0.83rem * 5)) / 6);
  height: 16rem;
  border: 0.02rem solid #e3e3e3;
  margin: 0 0.83rem 0.83rem 0;
  padding: 0.67rem;
  cursor: pointer;
  &:nth-child(6n) {
    margin-right: 0;
  }
  &:active,
  &:hover {
    background: #fff9f6;
    border: 0.04rem solid #ff7a3e;
  }
  .product-image {
    position: relative;
    width: 100%;
    height: 8rem;
    box-sizing: border-box;
    margin-bottom: 0.5rem;
    .image {
      width: 100%;
      height: 100%;
      object-fit: cover;
    }
  }
  .product-title {
    width: 100%;
    font-size: 13px;
    font-weight: 500;
    color: #333333;
    display: -webkit-box;
    -webkit-box-orient: vertical;
    -webkit-line-clamp: 2;
    overflow: hidden;
    margin-bottom: 0.33rem;
    min-height: 2rem;
  }
  .product-desc {
    position: relative;
    width: 100%;
    font-size: 12px;
    font-weight: 400;
    color: #999999;
    margin-bottom: 0.1rem;
    align-items: baseline;
    .label {
      width: 60%;
    }
    .text {
      width: 40%;
      text-align: right;
    }
    .money {
      font-size: 16px;
      font-weight: 500;
      color: #ff1b1b;
      .label {
        font-size: 12px;
        font-weight: bold;
      }
      .num {
        max-width: 60%;
      }
      .unit {
        margin-left: 0.68rem;
        font-size: 10px;
        font-weight: 400;
        color: #999999;
      }
    }
  }
  .product-store {
    position: absolute;
    left: 0.67rem;
    bottom: 0.67rem;
    .title {
      width: 6rem;
      font-size: 12px;
      font-weight: 400;
      color: #666666;
      margin-left: 0.25rem;
    }
  }
  .product-cart {
    position: absolute;
    right: 0.67rem;
    bottom: 0.67rem;
    width: 2rem;
    height: 2rem;
    background: #ff8b2e;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    &:active,
    &:hover {
      filter: brightness(0.9);
    }
  }
`;
