import { useState, useEffect, useContext } from 'react';

import { Button, Empty, Pagination, Spin } from 'antd';
import { useRouter } from 'next/router';

import { UserContext } from '~/lib/userProvider';
import { bigNumberTransform } from '~/utils/money';

import api, { Item } from './api';
import styles from './index.module.scss';

type Props = {
  params?: {
    cityCode: number; // 市编码
    date: string; // 时间 2023-05-19
    districtCode: number; // 区编码
    provinceCode: number; // 省编码
  };
  showWxCodeModalEvent: () => void;
};

export default function Bids(props: Props) {
  const router = useRouter();
  const [list, setList] = useState<Array<Item>>([]);
  const [pageParams, setPageParams] = useState({
    pageNo: 1,
    pageSize: 10,
  });
  const [count, setCount] = useState(0);
  const [abort, setAbort] = useState<AbortController | null>(null);
  const [loading, setLoading] = useState(false);
  const { userInfo, setNeedLogin } = useContext(UserContext);
  const [reload, setReload] = useState(false);

  useEffect(() => {
    // 中断前一次请求
    abort?.abort();
    setAbort(new AbortController());
  }, [pageParams, props.params, reload]);

  useEffect(() => {
    if (!abort) {
      return;
    }
    setLoading(true);
    api
      .listNewTenderInfo(
        {
          ...pageParams,
          ...props.params,
        },
        {
          signal: abort?.signal,
        },
      )
      .then((res) => {
        setList(res.result?.list || []);
        setCount(res.result?.totalCount || 0);
        setLoading(false);
      });
  }, [abort]);

  const onPageChange = (page: number, pageSize: number) => {
    setPageParams({
      ...pageParams,
      pageNo: page,
    });
  };

  // 申请合作
  const onApply = (item: Item) => {
    if (userInfo) {
      api
        .apply({
          tenderNewsId: item.tenderNewsId,
          tenderInfoId: item.id,
        })
        .then((res) => {
          if (res.code === '200') {
            window.messageApi.success('申请成功');
            setReload(!reload);
          }
        });
    } else {
      setNeedLogin(true);
    }
  };
  // 招标项目详情
  const toTenderProjectsDetail = (item: Item) => {
    router.push(`/projectInfo/tenderProjectsDetail/${item.id}`);
  };

  return (
    <Spin spinning={loading} delay={500}>
      <div className={styles.bids} style={{ height: 610 }}>
        {list.map((item) => {
          return (
            <div className={styles.item} key={item.id}>
              <div className={styles.info} onClick={() => toTenderProjectsDetail(item)}>
                <div className={styles.title}>{item.tenderTitle}</div>
              </div>
              <Button type='primary' className={styles.btn}>
                <div className={styles.text1}>
                  {bigNumberTransform(item.tenderPrice * 10000, true)}
                </div>
                <div className={styles.text2} onClick={() => props.showWxCodeModalEvent()}>
                  申请合作
                </div>
              </Button>
            </div>
          );
        })}
        {list.length === 0 && <Empty></Empty>}
      </div>
      <Pagination
        current={pageParams.pageNo}
        defaultPageSize={pageParams.pageSize}
        showSizeChanger={false}
        showQuickJumper
        total={count}
        onChange={onPageChange}
        hideOnSinglePage={true}
        style={{ marginTop: 20 }}
      />
    </Spin>
  );
}
