import React, { useEffect, useState } from 'react';
import { Checkbox } from 'antd';
import { CheckboxValueType } from 'antd/es/checkbox/Group';
import Big from 'big.js';
import { useSelector } from 'react-redux';
import styled from 'styled-components';
import { InterDataType } from '@/api/interface';
import { UserPayWalletInfoType } from '@/api/interface/wallet';
import { SystemState } from '@/store/module/system';
import { isNone } from '@/utils';
import getCurrentUserPayWalletInfo from '@/utils/getCurrentUserPayWalletInfo';
import { formatMoney } from '@/utils/money';

// 钱包类型
type WalletType = InterDataType<UserPayWalletInfoType>;

const PaymentCheckout: React.FC<{ cost: number }> = ({ cost }) => {
  // system
  const system = useSelector((state: any) => state.system) as SystemState;
  // 选择的支付方式
  const [paymentTypeValue, setPaymentTypeValue] = useState<CheckboxValueType[]>(
    [],
  );
  // 钱包详情
  const [walletInfo, setWalletInfo] = useState<WalletType>();
  // 获取钱包详情
  const getUserPayWalletInfo = async () => {
    const res = await getCurrentUserPayWalletInfo();
    if (res) setWalletInfo(res);
  };
  // 获取用户最多可抵扣的云享金余额
  const getOrderPriceCashMax = () => {
    // 当前的云享金余额
    const cash = walletInfo?.cashAmt || 0;
    // 如果用户的钱包金额大于等于订单金额 则可抵扣金额为订单金额
    return cash >= (cost || 0) ? cost || 0 : cash;
  };
  // 获取用户最多可抵扣的薪水余额
  const getOrderPriceSalaryMax = () => {
    // 当前的薪水余额
    const salary = walletInfo?.salaryAmt || 0;
    // 如果用户的钱包金额大于等于订单金额 则可抵扣金额为订单金额
    return salary >= (cost || 0) ? cost || 0 : salary;
  };
  // 获取用户最多可抵扣的总的金额
  const getOrderPriceTotalMax = () => {
    const cash = Big(getOrderPriceCashMax());
    const salary = Big(getOrderPriceSalaryMax());
    return cash.add(salary).toNumber();
  };
  // 获取用户的云享金余额
  const getOrderPriceCash = () => {
    // 云享金是否选中
    const selectCash = paymentTypeValue?.includes(1);
    // 余额是否选中
    const selectSalary = paymentTypeValue?.includes(2);
    // 总的抵扣金额
    const total =
      getOrderPriceTotalMax() > cost ? cost : getOrderPriceTotalMax();
    // 只选其中一个的情况
    if (!selectCash && selectSalary) {
      return Big(total).minus(Big(getOrderPriceSalaryMax())).toNumber();
    }
    // 返回结果
    return getOrderPriceCashMax();
  };
  // 获取用户的薪水余额
  const getOrderPriceSalary = () => {
    // 云享金是否选中
    const selectCash = paymentTypeValue?.includes(1);
    // 余额是否选中
    const selectSalary = paymentTypeValue?.includes(2);
    // 总的抵扣金额
    const total =
      getOrderPriceTotalMax() > cost ? cost : getOrderPriceTotalMax();
    // 只选其中一个的情况
    if (!selectCash && selectSalary) {
      return getOrderPriceSalaryMax();
    }
    // 返回结果
    return Big(total).minus(Big(getOrderPriceCashMax())).toNumber();
  };
  // 根据用户钱包和需要支付的金额设置默认选中
  const setDefaultSelect = () => {
    // 必须有金额时才继续执行
    if (isNone(walletInfo?.cashAmt) || isNone(walletInfo?.salaryAmt)) {
      return;
    }
    // 云享金
    const cash = Big(walletInfo?.cashAmt || 0).toNumber();
    // 余额
    const salary = Big(walletInfo?.salaryAmt || 0).toNumber();
    // 如果需要支付金额为零 则后面的判断都不执行
    if (!cost || cost <= 0) {
      setPaymentTypeValue([]);
      return;
    }
    // 如果云享金大于等于订单金额 则默认选中云享金
    if (cash >= cost) {
      setPaymentTypeValue([1]);
      return;
    }
    // 如果余额大于等于订单金额 则默认选中余额
    if (salary >= cost) {
      setPaymentTypeValue([2]);
      return;
    }
    // 如果云享金有钱
    if (cash > 0 && salary === 0) {
      setPaymentTypeValue([1]);
      return;
    }
    // 如果余额有钱
    if (cash === 0 && salary > 0) {
      setPaymentTypeValue([2]);
      return;
    }
    // 如果两者都有余额
    if (cash > 0 && salary > 0) {
      setPaymentTypeValue([1, 2]);
    }
  };
  // 组件挂载
  useEffect(() => {
    if (!system?.token) return;
    getUserPayWalletInfo().then();
    if (!cost) return;
    setDefaultSelect();
  }, [cost]);
  return (
    <PaymentCheckoutWrap>
      <Checkbox.Group value={paymentTypeValue} onChange={setPaymentTypeValue}>
        <div className="payment-item flex-start">
          <img
            src="https://file.iuav.com/file/checkout_icon_01.png"
            alt="云享金"
            className="icon"
          />
          <div className="title">云享金</div>
          <div className="num">-可抵扣￥{formatMoney(getOrderPriceCash())}</div>
          <div className="checkbox">
            <Checkbox value={1} />
          </div>
        </div>
        <div className="payment-item flex-start">
          <img
            src="https://file.iuav.com/file/checkout_icon_02.png"
            alt="余额"
            className="icon"
          />
          <div className="title">余额</div>
          <div className="num">
            -可抵扣￥{formatMoney(getOrderPriceSalary())}
          </div>
          <div className="checkbox">
            <Checkbox value={2} />
          </div>
        </div>
      </Checkbox.Group>
    </PaymentCheckoutWrap>
  );
};

export default PaymentCheckout;
// 样式
const PaymentCheckoutWrap = styled.div`
  position: relative;
  width: 18.6rem;
  .payment-item {
    position: relative;
    width: 100%;
    &:not(:last-child) {
      margin-bottom: 0.66rem;
    }
    .icon {
      width: 1.25rem;
      height: 1.25rem;
    }
    .title,
    .num {
      margin-left: 0.5rem;
    }
    .num {
      color: #ff4600;
    }
    .checkbox {
      position: absolute;
      top: 0;
      right: 0;
    }
  }
`;
