import React, { useEffect, useState } from 'react';
import { Empty } from 'antd';
import { InterDataType } from '@/api/interface';
import {
  GetAppCategoryInfo,
  QueryGoodsInfoByCategorySub,
} from '@/api/interface/mall';
import { MallAPI } from '@/api/modules/mall';
import BreadcrumbView from '@/components/breadcrumb';
import CategorySelectView, { CategoryType } from '@/components/categorySelect';
import LayoutView from '@/components/layout';
import ProductItemView from '@/components/productItem';
import ProductListView from '@/components/productList';
import { MallWrap } from '@/pages/mall/styled';

// 分类列表类型
type CategoryListType = InterDataType<GetAppCategoryInfo>;
// 商品列表类型
type GoodsInfoListType = InterDataType<QueryGoodsInfoByCategorySub>;

const MallView: React.FC<{
  categoryList: CategoryListType;
}> = (props) => {
  // 分类列表
  const [categoryList, setCategoryList] = useState<CategoryType>([]);
  // 商品列表
  const [goodsInfoList, setGoodsInfoList] = useState<GoodsInfoListType>([]);
  // 分页数据
  const [pagination, setPagination] = useState({
    pageNo: 1,
    pageSize: 18,
    totalCount: 0,
  });
  // 转换分类列表
  const getCategoryList = () => {
    setCategoryList(
      props?.categoryList
        ?.filter((i) => i.subDTOList?.length)
        ?.map((i) => ({
          value: i.id,
          label: i.name,
          children: i.subDTOList?.map((n) => ({ value: n.id, label: n.name })),
        })),
    );
  };
  // 获取商品列表
  const getGoodsInfoByCategorySub = async (list: number[]) => {
    const res = await MallAPI.queryGoodsInfoByCategorySub(list);
    if (res && res.code === '200') {
      setGoodsInfoList(res.result || []);
      setPagination({
        ...pagination,
        pageNo: 1,
        totalCount: res.result?.length || 0,
      });
      // console.log('商品列表 --->', res.result);
    }
  };
  // 根据分页数据返回商品列表
  const getGoodsInfoListByPage = () => {
    const { pageNo, pageSize, totalCount } = pagination;
    return goodsInfoList?.slice(
      (pageNo - 1) * pageSize,
      pageNo * (pageSize < totalCount ? pageSize : totalCount),
    );
  };
  // 翻页回调
  const handlePageChange = (pageNo: number, pageSize: number) => {
    setPagination({ ...pagination, pageNo, pageSize });
  };
  // 组件挂载
  useEffect(() => {
    if (!props) return;
    getCategoryList();
    // console.log('执行到此处12312 --->', props);
  }, [props]);
  return (
    <>
      <LayoutView placeholder={true}>
        <MallWrap>
          {/* 面包屑 */}
          <BreadcrumbView />
          {/* 类型筛选 */}
          <CategorySelectView
            list={categoryList}
            onSecond={getGoodsInfoByCategorySub}
          />
          {/* 产品列表 */}
          <ProductListView pagination={pagination} onChange={handlePageChange}>
            {getGoodsInfoListByPage()?.length ? (
              getGoodsInfoListByPage()?.map((i, j) => (
                <ProductItemView key={j} detail={i} />
              ))
            ) : (
              <div className="list-empty flex-center">
                <Empty />
              </div>
            )}
          </ProductListView>
        </MallWrap>
      </LayoutView>
    </>
  );
};

export default MallView;
