import React, { useEffect, useState } from 'react';
import { Empty } from 'antd';
import styled from 'styled-components';
import { InterListType, InterReqType } from '@/api/interface';
import {
  GetIndustryListPagesType,
  GetListAPPCompanyInspectionPageType,
} from '@/api/interface/service';
import { ServiceAPI } from '@/api/modules/service';
import BreadcrumbView from '@/components/breadcrumb';
import CategorySelectView, { CategoryType } from '@/components/categorySelect';
import LayoutView from '@/components/layout';
import ProductListView from '@/components/productList';
import ServiceItemView from '@/components/serviceItem';

// 分类列表类型
type CategoryListType = InterListType<GetIndustryListPagesType>;
// 请求参数类型
type ReqType = InterReqType<GetListAPPCompanyInspectionPageType>;
// 服务列表
type ListType = InterListType<GetListAPPCompanyInspectionPageType>;

const ServiceView: React.FC<{
  categoryList: CategoryListType;
}> = (props) => {
  // 分类列表
  const [categoryList, setCategoryList] = useState<CategoryType>([]);
  // 转换分类列表
  const getCategoryList = () => {
    setCategoryList(
      props?.categoryList
        ?.filter((i) => i?.inspectionDTOS?.length)
        ?.map((i) => ({
          value: i.id,
          label: i.typeName,
          children: i?.inspectionDTOS?.map((n) => ({
            value: n.id,
            label: n.inspectionName,
          })),
        })),
    );
  };
  // 分页数据
  const [pagination, setPagination] = useState({
    pageNo: 1,
    pageSize: 18,
    totalCount: 0,
  });
  // 服务列表
  const [inspectionList, setInspectionList] = useState<ListType>([]);
  // 获取服务列表
  const getListAPPCompanyInspectionPage = async (data: ReqType) => {
    const res = await ServiceAPI.getListAPPCompanyInspectionPage({
      pageNo: pagination.pageNo,
      pageSize: pagination.pageSize,
      ...data,
    });
    if (res && res.code === '200') {
      const { list, totalCount, pageNo, pageSize } = res.result;
      setInspectionList(list || []);
      setPagination({
        ...pagination,
        totalCount,
        pageNo,
        pageSize,
      });
    }
  };
  // 分类筛选
  const handleSelect = async (e: { main?: number; second?: number[] }) => {
    if (e.second?.length && e.second?.length > 1) {
      await getListAPPCompanyInspectionPage({
        industryTypeId: e?.main,
        pageNo: 1,
      });
    } else {
      await getListAPPCompanyInspectionPage({
        inspectionId: e?.second?.[0],
        pageNo: 1,
      });
    }
  };
  // 翻页回调
  const handlePageChange = async (pageNo: number, pageSize: number) => {
    await getListAPPCompanyInspectionPage({ pageNo, pageSize });
  };
  // 组件挂载
  useEffect(() => {
    if (!props) return;
    getCategoryList();
  }, [props]);
  return (
    <LayoutView>
      <ServiceWrap>
        {/* 面包屑 */}
        <BreadcrumbView />
        {/* 类型筛选 */}
        <CategorySelectView
          allText="全部服务"
          list={categoryList}
          isMultiple={false}
          onSelect={handleSelect}
        />
        {/* 产品列表 */}
        <ProductListView pagination={pagination} onChange={handlePageChange}>
          {inspectionList?.length ? (
            inspectionList?.map((i, j) => (
              <ServiceItemView key={j} detail={i} />
            ))
          ) : (
            <div className="list-empty flex-center">
              <Empty />
            </div>
          )}
        </ProductListView>
      </ServiceWrap>
    </LayoutView>
  );
};

export default ServiceView;
export const ServiceWrap = styled.div`
  position: relative;
  max-width: 1190px;
  box-sizing: border-box;
  padding: 2rem 0 0 0;
  margin: 0 auto;
`;
