import React, { useEffect, useState } from 'react';
import {
  ExportOutlined,
  LikeOutlined,
  MessageOutlined,
} from '@ant-design/icons';
import { App, Button, Image } from 'antd';
import { useRouter } from 'next/router';
import { useDispatch, useSelector } from 'react-redux';
import styled from 'styled-components';
import { HomeAPI } from '@/api';
import { InterListType } from '@/api/interface';
import { ForumListType } from '@/api/interface/home';
import { RootState } from '@/store';
import { setGlobalData } from '@/store/module/globalData';
import { SystemState } from '@/store/module/system';

// 详情类型
type DetailType = InterListType<ForumListType>[0];

const ForumItemView: React.FC<{
  detail: DetailType;
  isDetail?: Boolean;
  hiddenMedia?: Boolean;
  onRefresh?: () => void;
}> = ({ detail, isDetail = false, onRefresh, hiddenMedia = false }) => {
  // 静态组件
  const { message } = App.useApp();
  // 路由钩子
  const router = useRouter();
  // system
  const system = useSelector((state: RootState) => state.system) as SystemState;
  // store
  const dispatch = useDispatch();
  // 缓存贴子的数据
  const [forumDetail, setForumDetail] = useState<DetailType>();
  // 获取帖子的媒体信息 0图片 1视频
  const getForumMedia = (type: number) => {
    return detail?.mediaVO?.filter((i) => i?.type === type)?.slice(0, 4);
  };
  // 帖子相关操作
  const handleAction = (type: number) => {
    // 评论
    if (type === 2) {
      // 如果是在详情页面则不执行跳转
      if (isDetail) return;
      // 跳转到评论页
      dispatch(setGlobalData({ loadingSpinnerVisible: true }));
      router.push(`/forum/detail/${detail?.id}`).then();
    }
  };
  // 点赞
  const handleReviewLikes = async () => {
    // 如果未登录，则弹出登录框
    if (!system?.token) {
      dispatch(setGlobalData({ loginModalVisible: true }));
      return;
    }
    const res = await HomeAPI.likeOrCancel({
      dynamicId: detail?.id,
    });
    if (res && res.code === '200') {
      message.success(!forumDetail?.likes ? '点赞成功' : '取消点赞');
      setForumDetail({
        ...detail,
        likes: !forumDetail?.likes,
        likesCount: !forumDetail?.likes
          ? Number(forumDetail?.likesCount) + 1
          : Number(forumDetail?.likesCount) - 1,
      });
    }
  };
  // 监听变化
  useEffect(() => {
    if (!detail) return;
    setForumDetail(detail);
  }, [detail]);
  return (
    <ForumItemWrap>
      <div className="relative mb-2 flex w-full items-center justify-start">
        <img
          className="h-9 w-9 rounded-full"
          src={
            detail?.userBaseInfo?.userImg ||
            'https://file.iuav.com/file/sharefly-logo.png'
          }
          alt="头像"
        />
        <div className="ml-2">
          <div className="font-bold">
            {detail?.userBaseInfo?.nickName || '微信用户'}
          </div>
          <div className="text-aaa">{detail?.dynamicPublishTime}</div>
        </div>
      </div>
      <div className="mb-3 w-full leading-5 tracking-wide">
        {detail?.forumGambitDTOList?.map((i, j) => (
          <span key={j} className="mr-1 cursor-pointer font-bold text-tag">
            {i.gambitName}
          </span>
        ))}
        <span>{detail?.description}</span>
      </div>
      {!hiddenMedia && (
        <div className="forum-media flex w-full flex-wrap items-start justify-start">
          {getForumMedia(0)?.length
            ? getForumMedia(0)?.map((i, j) => (
                <Image
                  src={`${i?.url}?x-oss-process=image/quality,Q_50`}
                  alt="图片"
                  key={j}
                  fallback={
                    'https://pad-video-x.oss-cn-shenzhen.aliyuncs.com/file/identity-bg.png'
                  }
                />
              ))
            : undefined}
          {getForumMedia(1)?.length
            ? getForumMedia(1)?.map((i, j) => (
                <div
                  className="media-video relative box-border overflow-hidden rounded-lg"
                  key={j}
                >
                  {/* eslint-disable-next-line jsx-a11y/media-has-caption */}
                  <video src={i.url} controls={true} />
                </div>
              ))
            : undefined}
        </div>
      )}
      <div className="forum-action flex pb-2 text-777">
        <div className="action-item mr-6" onClick={() => handleAction(1)}>
          <Button
            type={'text'}
            icon={<LikeOutlined />}
            className={!forumDetail?.likes ? 'text-333' : 'text-primary'}
            onClick={handleReviewLikes}
          >
            &nbsp;{forumDetail?.likesCount}
          </Button>
        </div>
        <div className="action-item mr-6" onClick={() => handleAction(2)}>
          <Button type={'text'} icon={<MessageOutlined />}>
            &nbsp;{forumDetail?.commentCount}
          </Button>
        </div>
        <div className="action-item mr-6" onClick={() => handleAction(3)}>
          <Button type={'text'} icon={<ExportOutlined />}>
            &nbsp;{forumDetail?.transpond || 0}
          </Button>
        </div>
      </div>
    </ForumItemWrap>
  );
};

export default ForumItemView;
// 样式
const ForumItemWrap = styled.div`
  position: relative;
  width: 100%;
  box-sizing: border-box;
  margin-bottom: 1rem;
  .forum-media {
    .media-video {
      width: 500px;
      height: calc(500px / 16 * 9);
      margin-bottom: 1rem;
      video {
        width: 100%;
        height: 100%;
      }
    }
  }
  .forum-action {
    .action-item {
      cursor: pointer;
      &:hover {
        color: #ff552d;
      }
    }
  }
  .ant-image {
    width: calc((100% - 0.67rem * 3) / 4);
    height: 8.6rem;
    margin: 0 0.67rem 0.67rem 0;
    border-radius: 0.5rem;
    overflow: hidden;
    &:nth-child(4n) {
      margin-right: 0;
    }
    .ant-image-img {
      object-fit: cover;
      height: 8.6rem;
    }
  }
`;
