import React, { useContext, useEffect, useState } from 'react';

import { Avatar, Button, Dropdown, Space, Tabs } from 'antd';
import type { TabsProps } from 'antd';
import { useRouter } from 'next/router';

import LoginModal from '~/components/loginModal';
import { UserContext } from '~/lib/userProvider';

import styles from './index.module.scss';
import JoinModal from './joinModal';
import PublishModal from './publishModal';

const items: TabsProps['items'] = [
  {
    key: '/home',
    label: `  首页  `,
  },
  {
    key: '/mall',
    label: `产品商城`,
  },
  {
    key: '/equipmentLeasing',
    label: `设备租赁`,
  },
  {
    key: '/flyingHandService',
    label: `飞手培训`,
  },
  {
    key: '/jobServices',
    label: `作业服务`,
  },
  {
    key: '/projectInfo',
    label: `项目资讯`,
  },
  {
    key: '/forum',
    label: `社区论坛`,
  },
];

type Props = {
  style?: React.CSSProperties;
};

export default function NavHeader(props: Props) {
  const router = useRouter();
  const [currentPath, setCurrentPath] = useState('');
  const { userInfo, logout, setNeedLogin, needLogin } = useContext(UserContext);

  useEffect(() => {
    const routerTo = items?.filter((item) => router.route == item.key)[0];
    if (routerTo) {
      setCurrentPath(routerTo?.key!);
    } else {
      setCurrentPath(router.route);
    }
  }, [router.route]);

  // 导航更改
  const onChange = (key: string) => {
    router.push(key);
  };

  // 退出登录
  const onLogout = () => {
    logout();
  };

  const [openLoginModal, setOpenLoginModal] = useState(false); // 登录modal
  const [openPublishModal, setOpenPublishModal] = useState(false); // 发布modal
  const [openJoinModal, setOpenJoinModal] = useState(false); // 加盟modal

  // 发布按钮事件
  function onPublish() {
    // 登录判断
    if (!userInfo) {
      setOpenLoginModal(true);
    } else {
      setOpenPublishModal(true);
    }
  }

  // 加盟按钮事件
  function onJoin() {
    // 登录判断
    if (!userInfo) {
      setOpenLoginModal(true);
    } else {
      setOpenJoinModal(true);
    }
  }

  // 从其它组件通知需要登录
  useEffect(() => {
    if (needLogin) {
      setOpenLoginModal(true);
    }
  }, [needLogin]);

  return (
    <div className={styles.navHeader} style={props.style}>
      <div className={styles.navHeaderContent}>
        <div className={styles.nav}>
          <div className={styles.logo}></div>
          <Tabs
            className={styles.tabs}
            activeKey={currentPath}
            items={items}
            onChange={onChange}
            onTabClick={onChange}
          />
          <Space size={16} className={styles.btns}>
            <Button type='primary' className={styles.btn1} onClick={onPublish}>
              + 发布需求
            </Button>
            <Button className={styles.btn2} onClick={onJoin}>
              加盟入驻
            </Button>
          </Space>
          {userInfo ? (
            <div className={styles.haedImg}>
              <Dropdown
                menu={{
                  items: [
                    {
                      key: '2',
                      label: (
                        <div onClick={() => router.push('/personalCenter/servicesOrders')}>
                          我的订单
                        </div>
                      ),
                    },
                    { key: '1', label: <div onClick={onLogout}>退出登录</div> },
                  ],
                }}
              >
                <Avatar size={36} style={{ background: '#bdbdbd' }} src={userInfo.userImg}></Avatar>
              </Dropdown>
            </div>
          ) : (
            <Button
              type='text'
              onClick={() => setOpenLoginModal(true)}
              style={{ fontWeight: 'bold', fontSize: 16 }}
            >
              登录
            </Button>
          )}
        </div>
      </div>
      <LoginModal
        open={openLoginModal}
        onCancel={() => {
          setOpenLoginModal(false);
          setNeedLogin(false);
        }}
      ></LoginModal>
      <PublishModal
        open={openPublishModal}
        onCancel={() => {
          setOpenPublishModal(false);
        }}
      ></PublishModal>
      <JoinModal
        open={openJoinModal}
        onCancel={() => {
          setOpenJoinModal(false);
        }}
      ></JoinModal>
    </div>
  );
}
