import React, { useEffect, useState } from 'react';
import { App, Button, Checkbox, Col, Form, Input, Row } from 'antd';
import { useRouter } from 'next/router';
import { useDispatch, useSelector } from 'react-redux';
import styled from 'styled-components';
import { CommonAPI } from '@/api';
import { InterReqType } from '@/api/interface';
import { WebRegisterType } from '@/api/interface/common';
import LayoutView from '@/components/layout';
import { RootState } from '@/store';
import { setGlobalData } from '@/store/module/globalData';
import { SystemState } from '@/store/module/system';

// 表单类型
type ReqType = InterReqType<WebRegisterType>;

const UserSignupView = () => {
  // useApp
  const { message } = App.useApp();
  // store
  const dispatch = useDispatch();
  // system
  const system = useSelector((state: RootState) => state.system) as SystemState;
  // router
  const router = useRouter();
  // 表单钩子
  const [formRef] = Form.useForm<ReqType>();
  // 是否选择协议
  const [checkValue, setCheckValue] = useState<boolean>(false);
  // 验证码倒计时
  const [countDown, setCountDown] = useState<number>(60);
  // 是否可以点击获取验证码
  const [isCanGetCode, setIsCanGetCode] = useState<boolean>(true);
  // 提交数据
  const handleSubmit = async () => {
    if (!checkValue) {
      await message.warning('请先阅读并接受用户协议和隐私协议');
      return;
    }
    const values = await formRef.validateFields().catch((err) => {
      Promise.all(
        err?.errorFields?.map((i: { errors: string[] }) =>
          message.warning(i?.errors[0]),
        ),
      );
    });
    if (!values) return;
    message.loading('注册中...', 1);
    const res = await CommonAPI.webRegister(values);
    if (res && res.code === '200') {
      await message.success('注册成功');
      // 跳转回登录
      await router.replace('/');
      dispatch(setGlobalData({ loginModalVisible: true }));
    }
  };
  // 跳转回登录
  const handleLogin = () => {
    dispatch(setGlobalData({ loginModalVisible: true }));
  };
  // 获取二维码
  const handleGetCode = async () => {
    const values = await formRef.validateFields(['phoneNum']).catch((err) => {
      Promise.all(
        err?.errorFields?.map((i: { errors: string[] }) =>
          message.warning(i?.errors[0]),
        ),
      );
    });
    if (!values) return;
    // 禁止再次发送
    setIsCanGetCode(false);
    message.loading('发送中...', 1);
    const res = await CommonAPI.getVerifyCodeAuth({
      phoneNum: values.phoneNum,
    });
    // 提示
    if (res && res.code === '200') {
      message.success('验证码已发送');
    } else {
      message.success(res.message);
    }
    // 开始倒计时
    let count = 60;
    const timer = setInterval(() => {
      count -= 1;
      setCountDown(count);
      if (count <= 0) {
        clearInterval(timer);
        setIsCanGetCode(true);
      }
    }, 1000);
  };
  // 打开协议弹窗
  const handleAgree = (type: number) => {
    dispatch(setGlobalData({ agreeModalVisible: true, agreeModalType: type }));
  };
  // 页面挂载
  useEffect(() => {
    if (!system?.token) return;
    // 如果有token则跳转回首页
    window.location.href = '/';
  }, [system]);
  return (
    <LayoutView placeholder={false}>
      <UserSignupBg />
      <UserSignupWrap>
        <div className="login-title"></div>
        <div className="login-box">
          <div className="title">欢迎注册</div>
          <div className="already mb-14">
            <span>已有账号？</span>
            <span className="bold cursor-pointer" onClick={handleLogin}>
              登录
            </span>
          </div>
          <Form
            name="signForm"
            form={formRef}
            autoComplete="off"
            className="w-full"
            size="large"
            labelAlign="left"
            labelCol={{ span: 4 }}
          >
            <Form.Item
              label="手机号"
              name="phoneNum"
              required={false}
              rules={[
                { required: true, message: '请输入要注册的手机号' },
                // 请输入正确的手机号
                {
                  pattern: /^1[3456789]\d{9}$/,
                  message: '请输入正确的手机号',
                },
              ]}
            >
              <Input placeholder="请输入手机号" maxLength={11} allowClear />
            </Form.Item>
            <Form.Item
              label="密码"
              name="password"
              required={false}
              rules={[{ required: true, message: '请输入密码' }]}
            >
              <Input
                placeholder="请输入密码"
                type="password"
                maxLength={20}
                allowClear
                autoComplete="new-password"
              />
            </Form.Item>
            <Form.Item label="验证码">
              <Row gutter={8}>
                <Col span={15}>
                  <Form.Item
                    name="code"
                    noStyle
                    rules={[
                      {
                        required: true,
                        message: '请输入验证码',
                      },
                    ]}
                  >
                    <Input
                      placeholder="请输入验证码"
                      maxLength={8}
                      allowClear
                    />
                  </Form.Item>
                </Col>
                <Col span={8}>
                  <Button onClick={handleGetCode} disabled={!isCanGetCode}>
                    {isCanGetCode ? '获取验证码' : `再次获取(${countDown}s)`}
                  </Button>
                </Col>
              </Row>
            </Form.Item>
          </Form>
          <div
            className={`submit mt-14 cursor-pointer ${checkValue && 'active'}`}
            onClick={handleSubmit}
          >
            立即注册
          </div>
          <div className="checkbox mt-4">
            <Checkbox
              checked={checkValue}
              onChange={(e) => setCheckValue(e.target?.checked)}
            >
              <span>阅读并接受</span>
              <span
                className="bold cursor-pointer"
                onClick={() => handleAgree(2)}
              >
                云享飞用户协议
              </span>
              <span>和</span>
              <span
                className="bold cursor-pointer"
                onClick={() => handleAgree(1)}
              >
                隐私协议
              </span>
            </Checkbox>
          </div>
        </div>
      </UserSignupWrap>
    </LayoutView>
  );
};

export default UserSignupView;
// 样式
const UserSignupWrap = styled.div`
  position: relative;
  height: 99.8vh;
  max-width: 1190px;
  margin: 0 auto;
  display: flex;
  align-items: center;
  justify-content: flex-end;
  box-sizing: border-box;
  padding: 0 5% 0 0;
  .login-title {
    position: absolute;
    top: 15%;
    left: 0;
    width: 9.25rem;
    height: 2.08rem;
    background-image: url('https://file.iuav.com/file/sharefly-signup-title.png');
    background-size: cover;
  }
  .login-box {
    position: relative;
    width: 23.29rem;
    height: 28.5rem;
    background: rgba(255, 255, 255, 0.86);
    border-radius: 0.58rem;
    box-sizing: border-box;
    display: flex;
    flex-direction: column;
    justify-content: flex-start;
    align-items: center;
    padding: 1.83rem 1.83rem 0;
    .title {
      position: relative;
      width: 100%;
      font-size: 20px;
      font-weight: bold;
      color: #333333;
      letter-spacing: 1px;
      margin-bottom: 0.2rem;
    }
    .already {
      position: relative;
      width: 100%;
      color: #999999;
      .bold {
        color: #fd562c;
      }
    }
    .submit {
      position: relative;
      box-sizing: border-box;
      width: 19.38rem;
      height: 2.5rem;
      background: #ffab95;
      border-radius: 1.29rem;
      text-align: center;
      line-height: 2.25rem;
      color: #ffffff;
    }
    .active {
      background: #fd562c;
    }
    .checkbox {
      .bold {
        color: #fd562c;
        margin: 0 0.25rem;
      }
    }
  }
  @media screen and (max-width: 1200px) {
    padding: 0;
    justify-content: center;
  }
  @media screen and (min-width: 1550px) {
    padding: 0;
  }
`;
// 图片
const UserSignupBg = styled.div`
  position: absolute;
  top: 0;
  left: 0;
  width: 100%;
  height: 99.8vh;
  background-image: url('https://file.iuav.com/file/sharefly-signup-bg.png');
  background-size: cover;
`;
