import React, { useEffect, useState } from 'react';
import { Carousel } from 'antd';
import { useRouter } from 'next/router';
import styled from 'styled-components';
import { InterDataType } from '@/api/interface';
import { QueryBrandGoodsType } from '@/api/interface/mall';

// 商品列表类型
type GoodsListType = InterDataType<QueryBrandGoodsType>;

const StoreProductRecommendView: React.FC<{
  goodsInfoList: GoodsListType;
}> = ({ goodsInfoList }) => {
  // 路由钩子
  const router = useRouter();
  // 推荐商品列表
  const [recommendList, setRecommendList] = useState<GoodsListType[0][][]>();
  // 获取最低价格
  const getLowerPrice = (item: GoodsListType[0]) => {
    const price =
      item.priceStock?.reduce((a: any, b: any) =>
        a.salePrice < b.salePrice ? a : b,
      ).salePrice || 0;
    return price.toLocaleString();
  };
  // 跳转商品详情
  const handleDetail = (detail: GoodsListType[0]) => {
    router.push(`/mall/product/${detail?.id}`).then();
  };
  // 组件挂载
  useEffect(() => {
    if (!goodsInfoList?.length) return;
    const list = goodsInfoList?.filter((i) => i?.recommend);
    setRecommendList(
      Array.from({ length: Math.ceil(list.length / 4) }, (_, index) =>
        list.slice(index * 4, (index + 1) * 4),
      ),
    );
  }, [goodsInfoList]);
  return recommendList?.length ? (
    <StoreProductRecommendWrap className="flex-start">
      <div className="recommend-title">店铺推荐</div>
      <div className="recommend-swiper">
        <Carousel autoplay={true} autoplaySpeed={8000}>
          {recommendList?.map((i, j) => (
            <div className="swiper-item flex-between" key={j}>
              {i?.map((n, m) => (
                <div
                  className="item-goods"
                  key={m}
                  onClick={() => handleDetail(n)}
                >
                  <img
                    className="image"
                    src={`${n?.resourcesList?.[0]?.url}?x-oss-process=image/quality,q_20`}
                    alt={n?.tradeName}
                  />
                  <div className="detail flex-between">
                    <div className="content">
                      <div className="title text-ellipsis">{n?.tradeName}</div>
                      <div className="text text-ellipsis">成交{n?.id}件</div>
                    </div>
                    <div className="price">
                      {n?.priceShow ? (
                        <>
                          <span className="label">￥</span>
                          <span className="num">{getLowerPrice(n)}</span>
                        </>
                      ) : (
                        <span className="label">咨询报价</span>
                      )}
                    </div>
                  </div>
                </div>
              ))}
            </div>
          ))}
        </Carousel>
      </div>
    </StoreProductRecommendWrap>
  ) : null;
};

export default StoreProductRecommendView;
// 样式
const StoreProductRecommendWrap = styled.div`
  position: relative;
  width: 100%;
  height: 28.6rem;
  box-sizing: border-box;
  //background: #9f9f9f;
  background-image: url('https://file.iuav.com/file/sharefly-store-product-bg.png');
  background-size: 100% 100%;
  flex-direction: column;
  padding-top: 1rem;
  .recommend-title {
    font-weight: 500;
    font-size: 28px;
    color: #ffffff;
    letter-spacing: 4px;
    margin-bottom: 2rem;
  }
  .recommend-swiper {
    position: relative;
    max-width: 1190px;
    box-sizing: border-box;
    .swiper-item {
      position: relative;
      width: 100%;
      height: 21rem;
      box-sizing: border-box;
      padding: 0 6rem 1rem 6em;
      display: flex !important;
      .item-goods {
        position: relative;
        width: calc((100% / 4) - 2rem);
        height: 18rem;
        background: #000;
        cursor: pointer;
        user-select: none;
        &:hover {
          transform: scale(1.1);
          transition: all ease-in-out 350ms;
        }
        .image {
          position: relative;
          width: 100%;
          height: 14rem;
          background: #ffffff;
          object-fit: cover;
        }
        .detail {
          position: relative;
          width: 100%;
          height: calc(100% - 14rem);
          padding: 0 1rem;
          box-sizing: border-box;
          .content {
            position: relative;
            width: 50%;
            .title {
              font-weight: 400;
              font-size: 14px;
              color: #ffffff;
            }
            .text {
              font-weight: 400;
              font-size: 13px;
              color: rgba(255, 255, 255, 0.8);
            }
          }
          .price {
            position: relative;
            width: 50%;
            text-align: right;
            font-weight: 500;
            font-size: 16px;
            color: #fd4034;
            .num {
              font-size: 18px;
            }
          }
        }
      }
    }
  }
`;
