import React, { useEffect, useState } from 'react';
import { Empty, Segmented, SegmentedProps } from 'antd';
import styled from 'styled-components';
import { HomeAPI } from '@/api';
import { InterDataType, InterListType } from '@/api/interface';
import { AppPublishListType } from '@/api/interface/home';
import { GetIndustryListPagesType } from '@/api/interface/service';
import { ServiceAPI } from '@/api/modules/service';
import LayoutView from '@/components/layout';
import ProductListView from '@/components/productList';
import ServiceTaskItem from '@/pages/service/task/comp/_serviceTaskItem';

// 列表类型
type ListType = InterDataType<AppPublishListType>;
// 行业类型
type IndustryType = InterListType<GetIndustryListPagesType>;

// 任务列表筛选
const actionList = [
  {
    label: '附近',
    value: 0,
    data: {
      cityCode: '330111',
      districtCode: '330100',
      provinceCode: '330000',
    },
  },
  { label: '最新', value: 1, data: { isNewRequirements: true } },
  { label: '高佣', value: 2, data: { isHighCommission: true } },
];

const ServiceTaskListPage = () => {
  // 行业列表
  const [industryList, setIndustryList] = useState<IndustryType>([]);
  // 分页数据
  const [pagination, setPagination] = useState({
    pageNo: 1,
    pageSize: 16,
    totalCount: 0,
  });
  // 实时订单列表
  const [requireList, setRequireList] = useState<ListType>([]);
  // 获取实时订单列表
  const getAppPublishList = async (index = 0) => {
    const res = await HomeAPI.appPublishList({
      ...actionList[index].data,
    });
    if (res && res.code === '200') {
      const list = res.result?.filter((i) => i?.publish) || [];
      setRequireList(list);
      setPagination({
        ...pagination,
        pageNo: 1,
        totalCount: list?.length || 0,
      });
    }
  };
  // 根据分页数据返回商品列表
  const getPaginationList = () => {
    const { pageNo, pageSize, totalCount } = pagination;
    return requireList?.slice(
      (pageNo - 1) * pageSize,
      pageNo * (pageSize < totalCount ? pageSize : totalCount),
    );
  };
  // 翻页回调
  const handlePageChange = (pageNo: number, pageSize: number) => {
    setPagination({ ...pagination, pageNo, pageSize });
  };
  // 获取各个目录及分类信息
  const getIndustryListPages = async () => {
    const res = await ServiceAPI.getIndustryListPages({
      pageNo: 1,
      pageSize: 999,
    });
    if (res && res.code === '200') {
      setIndustryList(res?.result?.list || []);
    }
  };
  // 筛选数据
  const handleSelectAction: SegmentedProps['onChange'] = (e) => {
    const index = actionList?.find((i) => i?.label === e)?.value || 0;
    getAppPublishList(index).then();
  };
  // 组件挂载
  useEffect(() => {
    Promise.all([getAppPublishList(), getIndustryListPages()]).then();
  }, []);
  return (
    <LayoutView contentTitle={'抢单大厅'}>
      <ServiceTaskListWrap>
        <div className="tab">
          <Segmented
            options={actionList?.map((i) => i?.label)}
            size="large"
            defaultValue={'附近'}
            onChange={handleSelectAction}
          />
        </div>
        {/* 产品列表 */}
        <ProductListView pagination={pagination} onChange={handlePageChange}>
          {getPaginationList()?.length ? (
            getPaginationList()?.map((i, j) => (
              <ServiceTaskItem key={j} detail={i} industryList={industryList} />
            ))
          ) : (
            <div className="list-empty flex-center">
              <Empty />
            </div>
          )}
        </ProductListView>
      </ServiceTaskListWrap>
    </LayoutView>
  );
};

export default ServiceTaskListPage;
// 样式
const ServiceTaskListWrap = styled.div`
  position: relative;
  max-width: 1190px;
  box-sizing: border-box;
  padding: 1rem 0 0 0;
  margin: 0 auto;
  .tab {
    position: relative;
    width: 100%;
    display: flex;
    justify-content: flex-end;
    margin-bottom: 1rem;
  }
`;
