import React, { useEffect, useState } from 'react';
import { Empty, Tabs, TabsProps } from 'antd';
import styled from 'styled-components';
import { HomeAPI } from '@/api';
import { InterListType } from '@/api/interface';
import { ListNewsType, ListTenderInfoType } from '@/api/interface/home';
import BreadcrumbView from '@/components/breadcrumb';
import LayoutView from '@/components/layout';
import ProductListView from '@/components/productList';
import NewsListItem from '@/pages/news/comp/_newsListItem';
import NewsSwiperView from '@/pages/news/comp/_newsSwiper';
import NewsTenderItem from '@/pages/news/comp/_newsTenderItem';

// 新闻列表类型
type NewsListType = InterListType<ListNewsType>;
// 招投标列表类型
type TenderListType = InterListType<ListTenderInfoType>;
// tab列表
const items: TabsProps['items'] = [
  {
    key: '1',
    label: '热点新闻',
  },
  {
    key: '2',
    label: '招标快讯',
  },
];

const NewsListView = () => {
  // tab栏切换
  const [tabIndex, setTabIndex] = useState('1');
  // 新闻列表
  const [newsList, setNewsList] = useState<NewsListType>([]);
  // 招投标列表
  const [tenderList, setTenderList] = useState<TenderListType>([]);
  // 分页数据
  const [pagination, setPagination] = useState({
    pageNo: 1,
    pageSize: 5,
    totalPage: 0,
    totalCount: 0,
  });
  // 热点资讯列表
  const getListNews = async (data?: { pageNo?: number; pageSize?: number }) => {
    const res = await HomeAPI.getListNews({
      pageNo: pagination.pageNo,
      pageSize: pagination.pageSize,
      ...data,
    });
    if (res && res.code === '200') {
      const { list, pageNo, pageSize, totalPage, totalCount } = res.result;
      setPagination({
        pageNo,
        pageSize,
        totalPage,
        totalCount,
      });
      // setNewsList((prevList) => [...prevList, ...(list || [])]);
      pagination.pageNo = pageNo;
      pagination.totalPage = totalPage;
      setNewsList(list || []);
    }
  };
  // 获取招标快讯列表
  const getListTenderInfo = async (data?: {
    pageNo?: number;
    pageSize?: number;
  }) => {
    const res = await HomeAPI.getListTenderInfo({
      pageNo: pagination.pageNo,
      pageSize: pagination.pageSize,
      ...data,
    });
    if (res && res.code === '200') {
      const { list, pageNo, pageSize, totalPage, totalCount } = res.result;
      setPagination({
        pageNo,
        pageSize,
        totalPage,
        totalCount,
      });
      // setNewsList((prevList) => [...prevList, ...(list || [])]);
      pagination.pageNo = pageNo;
      pagination.totalPage = totalPage;
      setTenderList(list || []);
    }
  };
  // tab栏切换监听
  const handleTabChange: TabsProps['onChange'] = async (e) => {
    setTabIndex(e);
    // pagination.pageNo = 1;
    // pagination.totalPage = 0;
    // pagination.totalCount = 0;
    setPagination({
      ...pagination,
      pageNo: 1,
      totalPage: 0,
      totalCount: 0,
    });
    setNewsList([]);
    if (e === '1') {
      await getListNews({ pageNo: 1 });
    } else {
      await getListTenderInfo({ pageNo: 1 });
    }
  };
  // 添加 handleReachBottom 函数
  const handlePageChange = async (pageNo: number, pageSize: number) => {
    if (tabIndex === '1') {
      await getListNews({ pageNo, pageSize });
    } else {
      await getListTenderInfo({ pageNo, pageSize });
    }
  };
  // 组件挂载
  useEffect(() => {
    getListNews().then();
  }, []);
  return (
    <LayoutView background={'#f9f9f9'}>
      <NewsListWrap>
        {/* 面包屑 */}
        <BreadcrumbView />
        {/* 轮播图 */}
        <NewsSwiperView />
        {/* 文章类型 */}
        <div className="news-tab">
          <Tabs
            defaultActiveKey={tabIndex}
            items={items}
            size={'large'}
            onChange={handleTabChange}
          />
        </div>
        {/* 文章列表 */}
        <div className="news-list">
          <ProductListView pagination={pagination} onChange={handlePageChange}>
            {tabIndex === '1' &&
              (newsList?.length ? (
                newsList?.map((i, j) => <NewsListItem key={j} detail={i} />)
              ) : (
                <div className="list-empty flex-center">
                  <Empty />
                </div>
              ))}
            {tabIndex === '2' &&
              (tenderList?.length ? (
                tenderList?.map((i, j) => <NewsTenderItem key={j} detail={i} />)
              ) : (
                <div className="list-empty flex-center">
                  <Empty />
                </div>
              ))}
          </ProductListView>
        </div>
      </NewsListWrap>
    </LayoutView>
  );
};

export default NewsListView;
// 样式
const NewsListWrap = styled.div`
  position: relative;
  max-width: 1190px;
  box-sizing: border-box;
  padding: 2rem 0 0 0;
  margin: 0 auto;
  .news-tab {
    position: relative;
    width: 100%;
    display: flex;
    justify-content: center;
    align-items: center;
    transform: translateY(-0.5rem);
  }
`;
