import React from 'react';
import { ShopFilled, ShoppingCartOutlined } from '@ant-design/icons';
import { useRouter } from 'next/router';
import styled from 'styled-components';
import { InterDataType } from '@/api/interface';
import { QueryGoodsInfoByCategorySub } from '@/api/interface/mall';

// 商品详情类型
type GoodsInfoListType = InterDataType<QueryGoodsInfoByCategorySub>[0];

const ProductItemView: React.FC<{
  detail: GoodsInfoListType;
  row?: number;
}> = ({ detail, row = 6 }) => {
  // 路由钩子
  const router = useRouter();
  // 获取最低价格
  const getLowerPrice = (item: GoodsInfoListType) => {
    const price =
      item.priceStock?.reduce((a: any, b: any) =>
        a.salePrice < b.salePrice ? a : b,
      ).salePrice || 0;
    return price.toLocaleString();
  };
  // 跳转商品详情
  const handleDetail = () => {
    router.push(`/mall/product/${detail?.id}`).then();
  };
  return (
    <ProductItemWrap onClick={handleDetail} $row={row}>
      <div className="product-image">
        <img
          className="image"
          src={`${detail?.resourcesList?.at(0)
            ?.url}?x-oss-process=image/quality,q_20`}
          alt={detail?.tradeName}
        />
      </div>
      <div className="product-title">{detail?.tradeName}</div>
      <div className="product-desc flex-between">
        <div className="label text-ellipsis">{detail?.description}</div>
        <div className="text text-ellipsis">成交{detail?.id}件</div>
      </div>
      <div className="product-money">
        {detail?.priceShow ? (
          <>
            <span className="label">￥</span>
            <span className="num">{getLowerPrice(detail)}</span>
          </>
        ) : (
          <span className="label">咨询报价</span>
        )}
      </div>
      <div className="product-store flex-start">
        <ShopFilled style={{ color: '#FF552D' }} />
        <div className="title text-ellipsis" title={detail?.companyName}>
          {detail?.companyName}
        </div>
      </div>
      <div className="product-cart">
        <ShoppingCartOutlined style={{ color: '#ffffff', fontSize: '16px' }} />
      </div>
    </ProductItemWrap>
  );
};

export default ProductItemView;
const ProductItemWrap = styled.div<{ $row: number }>`
  position: relative;
  box-sizing: border-box;
  width: calc(
    (100% - (0.83rem * ${(props) => props.$row - 1})) / ${(props) => props.$row}
  );
  height: 16.6rem;
  border: 0.02rem solid #e3e3e3;
  margin: 0 0.83rem 0.83rem 0;
  padding: 0.67rem;
  cursor: pointer;
  &:nth-child(${(props) => props.$row}n) {
    margin-right: 0;
  }
  &:active,
  &:hover {
    background: #fff9f6;
    border: 0.04rem solid #ff7a3e;
  }
  .product-image {
    position: relative;
    width: 100%;
    height: 8rem;
    box-sizing: border-box;
    margin-bottom: 0.5rem;
    .image {
      width: 100%;
      height: 100%;
      object-fit: contain;
    }
  }
  .product-title {
    width: 100%;
    font-size: 13px;
    font-weight: 500;
    color: #333333;
    display: -webkit-box;
    -webkit-box-orient: vertical;
    -webkit-line-clamp: 2;
    overflow: hidden;
    margin-bottom: 0.33rem;
    line-height: 16px;
    height: 32px;
  }
  .product-desc {
    position: relative;
    width: 100%;
    font-size: 12px;
    font-weight: 400;
    color: #999999;
    margin-bottom: 0.1rem;
    .label {
      width: 60%;
    }
    .text {
      width: 40%;
      text-align: right;
    }
  }
  .product-money {
    font-size: 16px;
    font-weight: 500;
    color: #ff1b1b;
    .label {
      font-size: 12px;
      font-weight: bold;
    }
  }
  .product-store {
    position: absolute;
    left: 0.67rem;
    bottom: 0.67rem;
    .title {
      width: 6rem;
      font-size: 12px;
      font-weight: 400;
      color: #666666;
      margin-left: 0.25rem;
    }
  }
  .product-cart {
    position: absolute;
    right: 0.67rem;
    bottom: 0.67rem;
    width: 2rem;
    height: 2rem;
    background: #ff8b2e;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    &:active,
    &:hover {
      filter: brightness(0.9);
    }
  }
  @media (prefers-color-scheme: dark) {
    .product-title {
      color: #fff;
    }
  }
`;
