import React, { useEffect, useState } from 'react';
import { ReloadOutlined, SearchOutlined } from '@ant-design/icons';
import {
  App,
  Button,
  Cascader,
  Form,
  Input,
  InputProps,
  Select,
  SelectProps,
  Space,
} from 'antd';
import styled from 'styled-components';

// 搜索类型
export type SearchColumns = {
  type: 'Input' | 'Select' | 'Cascader' | 'DatePicker' | 'RangePicker';
  label?: string;
  name: string;
  placeholder: string;
  maxlength?: number;
  width?: number;
  disable?: boolean;
  required?: boolean;
  options?: SelectProps['options'];
  onSelect?: (e: SelectProps['onSelect']) => void;
  onChange?: (e: InputProps['onChange']) => void;
  api?: any;
  transform?: (e: any) => SelectProps['options'];
};
// 组件传参
type PropsType = {
  columns?: SearchColumns[];
  onSearch?: (e: any) => void;
  preFixChild?: React.ReactNode;
  sufFixChild?: React.ReactNode;
  children?: React.ReactNode;
  isReset?: boolean;
  isSearch?: boolean;
  initialValues?: any;
  // onRef?: any;
};

const SearchBoxView: React.FC<PropsType> = ({
  columns,
  initialValues,
  onSearch,
  isReset = true,
  isSearch = true,
  sufFixChild,
}) => {
  // app
  const { message } = App.useApp();
  // 表单钩子
  const [formRef] = Form.useForm();
  // 提交数据
  const handleSubmit = async () => {
    const values = await formRef.validateFields().catch((e) => {
      const item = e.errorFields?.[0];
      message.warning(item?.errors?.[0]);
    });
    if (!values) return;
    onSearch?.(values);
  };
  // 重置数据
  const handleReset = () => {
    formRef.resetFields();
    onSearch?.({});
  };
  // 组件的optionList
  const [optionList, setOptionList] = useState<SelectProps['options'][]>();
  // 获取组件的option
  const getOptionData = async () => {
    // 如果没有api，直接返回
    if (!columns?.some((i) => i?.api)) return;
    // 获取数据
    const dataList = await Promise.all(
      // @ts-ignore es-lint-disable-next-line
      columns?.map((i) => i?.api),
    );
    // 设置数据
    setOptionList(
      columns?.map((i, j) => {
        if (i?.api) {
          return (
            i?.transform?.(dataList?.[j]?.result) || dataList?.[j]?.result || []
          );
        }
        return i?.options || [];
      }),
    );
  };
  // 组件挂载
  useEffect(() => {
    // if (!columns?.length) return;
    getOptionData().then();
  }, []);
  return (
    <SearchBoxWrap>
      <Form
        form={formRef}
        name="formRef"
        // labelCol={{ span: 8 }}
        // wrapperCol={{ span: 16 }}
        style={{ width: '100%' }}
        initialValues={initialValues}
        layout={'inline'}
        // onFinish={handleFinish}
        autoComplete="off"
        // onChange={handleChange}
      >
        {columns?.map((i, j) => (
          <Form.Item
            label={i?.label}
            name={i?.name}
            key={j}
            rules={[
              {
                required: i?.required || false,
                message: i?.label ? `请输入${i?.label}` : i?.placeholder,
              },
            ]}
          >
            {i?.type === 'Input' && (
              <Input
                placeholder={i?.placeholder}
                maxLength={i?.maxlength || 20}
                style={{ width: i?.width || '200px' }}
                disabled={i?.disable}
                allowClear
              />
            )}
            {i?.type === 'Select' && (
              <Select
                placeholder={i?.placeholder}
                style={{ width: i?.width || '200px' }}
                disabled={i?.disable}
                onSelect={i?.onSelect}
                options={optionList?.[j] || i?.options || []}
                allowClear
              />
            )}
            {i?.type === 'Cascader' && (
              <Cascader
                placeholder={i?.placeholder}
                style={{ width: i?.width || '200px' }}
                disabled={i?.disable}
                options={optionList?.[j] || i?.options || []}
                allowClear
              />
            )}
          </Form.Item>
        ))}
        <Form.Item>
          <Space>
            {isSearch && (
              <Button
                type="primary"
                icon={<SearchOutlined />}
                onClick={handleSubmit}
              >
                搜索
              </Button>
            )}
            {isReset && (
              <Button
                type="default"
                icon={<ReloadOutlined />}
                onClick={handleReset}
              >
                重置
              </Button>
            )}
          </Space>
        </Form.Item>
      </Form>
      {sufFixChild && <div className="suf-fix-child">{sufFixChild}</div>}
    </SearchBoxWrap>
  );
};

export default SearchBoxView;
// 样式
const SearchBoxWrap = styled.div`
  position: relative;
  width: 100%;
  box-sizing: border-box;
  display: flex;
  justify-content: flex-start;
  align-items: flex-start;
`;
