import React, { useEffect, useState } from 'react';
import { Empty } from 'antd';
import styled from 'styled-components';
import { HomeAPI } from '@/api';
import { InterListType, InterReqType } from '@/api/interface';
import { AppListPilotType } from '@/api/interface/home';
import BreadcrumbView from '@/components/breadcrumb';
import LayoutView from '@/components/layout';
import ProductListView from '@/components/productList';
import SearchBoxView, { SearchColumns } from '@/components/SearchBox';
import FlyerListItem from '@/pages/flyer/comp/_flyerListItem';

// 列表类型
type ListType = InterListType<AppListPilotType>;
// 参数类型
type ReqType = InterReqType<AppListPilotType>;
// 搜索列数据
const searchColumns: SearchColumns[] = [
  {
    type: 'Cascader',
    // label: '飞手姓名',
    name: 'areaNumber',
    placeholder: '请选择飞手地区',
    maxlength: 20,
    width: 120,
    api: HomeAPI.getSecondDistrictInfo(),
    transform: (data) =>
      data?.map((i: any) => ({
        label: i.name,
        value: i.id,
        children: i.childInfo?.map((k: any) => ({
          label: k.name,
          value: k.id,
        })),
      })),
  },
  {
    type: 'Select',
    // label: '任务能力',
    name: 'abilityId',
    placeholder: '请选择任务能力',
    width: 120,
    api: HomeAPI.IndustryListPages({ pageNo: 1, pageSize: 999 }),
    transform: (data) =>
      data.list?.map((i: any) => ({ label: i.typeName, value: i.id })),
  },
  {
    type: 'Select',
    // label: '执照类型',
    name: 'licenseType',
    placeholder: '请选择执照类型',
    width: 120,
    options: [
      { label: 'CAAC', value: 'CAAC' },
      { label: 'UTC', value: 'UTC' },
      { label: '其他', value: 'OTHER' },
    ],
  },
];
const FlyerPageView = () => {
  // 分页数据
  const [pagination, setPagination] = useState({
    pageNo: 1,
    pageSize: 16,
    totalPage: 0,
    totalCount: 0,
  });
  // 飞手列表
  const [flyerList, setFlyerList] = useState<ListType>([]);
  // 获取飞手列表
  const getFlyerList = async (data?: ReqType) => {
    const res = await HomeAPI.appListPilot({
      // 审批通过
      auditStatus: 1,
      pageNo: pagination.pageNo,
      pageSize: pagination.pageSize,
      ...data,
    });
    if (res && res.code === '200') {
      const { list, pageNo, totalPage, totalCount } = res.result;
      setFlyerList(list);
      setPagination({
        ...pagination,
        pageNo,
        totalPage,
        totalCount,
      });
      // console.log('列表数据 --->', list, pageNo, totalPage);
    }
  };
  // 搜索事件
  const handleSearch = async (value: ReqType) => {
    await getFlyerList({
      ...value,
      areaNumber: (value?.areaNumber as unknown as number[])?.at(-1),
      pageNo: 1,
    });
  };
  // 翻页事件
  const handlePageChange = async (pageNo: number, pageSize: number) => {
    await getFlyerList({ pageNo, pageSize });
  };
  // 组件挂载
  useEffect(() => {
    getFlyerList().then();
  }, []);
  return (
    <LayoutView>
      <FlyerPageWrap>
        {/* 面包屑 */}
        <BreadcrumbView />
        {/* 条件筛选 */}
        <SearchBoxView columns={searchColumns} onSearch={handleSearch} />
        <div className="h-4 w-full" />
        {/* 飞手列表 */}
        <ProductListView pagination={pagination} onChange={handlePageChange}>
          {flyerList?.length ? (
            flyerList?.map((i, j) => <FlyerListItem key={j} detail={i} />)
          ) : (
            <div className="list-empty flex-center">
              <Empty />
            </div>
          )}
        </ProductListView>
      </FlyerPageWrap>
    </LayoutView>
  );
};

export default FlyerPageView;
// 样式
const FlyerPageWrap = styled.div`
  position: relative;
  max-width: 1190px;
  box-sizing: border-box;
  padding: 2rem 0 0 0;
  margin: 0 auto;
`;
