import React, { useEffect, useState } from 'react';
import { Menu, MenuProps } from 'antd';
import { useRouter } from 'next/router';
import { InterDataType, InterListType } from '@/api/interface';
import {
  CourseDetailType,
  QueryCurriculumInfoListType,
  SelectCurriculumClassifyType,
} from '@/api/interface/course';
import { CourseAPI } from '@/api/modules/course';

// 详情类型
type DetailType = InterDataType<CourseDetailType>;
// 分类类型
type TabType = Array<
  InterDataType<SelectCurriculumClassifyType>[0] & {
    children?: InterDataType<SelectCurriculumClassifyType>;
  }
>;
// 列表类型
type ListType = InterListType<QueryCurriculumInfoListType>;

const CourseMenuView: React.FC<{ detail: DetailType }> = ({ detail }) => {
  // 路由钩子
  const router = useRouter();
  // 分类列表
  const [tabList, setTabList] = useState<TabType>([]);
  // 课程列表
  const [courseList, setCourseList] = useState<ListType>([]);
  // 获取课程分类
  const getCurriculumClassify = async () => {
    const res = await CourseAPI.selectCurriculumClassify();
    if (res && res.code === '200') {
      setTabList(
        res.result
          ?.filter((i) => !i.twoCourseId)
          ?.map((i) => {
            const children = res.result?.filter(
              (k) => k.oneCourseId === i.oneCourseId && k.twoCourseId,
            );
            return {
              ...i,
              children: children?.length ? children : undefined,
            };
          }),
      );
    }
  };
  // 获取课程列表
  const getCurriculumInfoList = async () => {
    const res = await CourseAPI.getCourseVideoList({
      pageNo: 1,
      pageSize: 999,
      oneCourseId: detail?.oneCourseId,
      twoCourseId: detail?.twoCourseId,
    });
    if (res && res.code === '200') {
      setCourseList(res.result.list || []);
    }
  };
  // 获取当前的分类信息
  const getCurrentMenu = (): MenuProps['items'] => {
    const main = tabList?.find((i) => i.oneCourseId === detail?.oneCourseId);
    const second = main?.children?.find(
      (i) => i.twoCourseId === detail?.twoCourseId,
    );
    return [
      {
        label: String(second?.name),
        key: String(second?.oneCourseId),
        children: courseList?.map((i) => ({
          label: i.curriculumName,
          key: String(i.id),
        })),
      },
    ];
  };
  // 切换课程
  const handleSelect: MenuProps['onSelect'] = (e) => {
    router.push(`/course/detail/${e.key}`).then();
  };
  // 组件挂载
  useEffect(() => {
    if (!detail?.id) return;
    Promise.all([getCurriculumClassify(), getCurriculumInfoList()]).then();
  }, [detail]);
  // 视图组件
  return (
    <div className="relative w-full">
      <Menu
        onSelect={handleSelect}
        style={{ width: '100%' }}
        mode="inline"
        items={getCurrentMenu()}
        theme="dark"
        defaultOpenKeys={[String(detail?.oneCourseId)]}
        defaultSelectedKeys={[String(detail?.id)]}
      />
    </div>
  );
};

export default CourseMenuView;
