import React, { useEffect, useState } from 'react';
import Big from 'big.js';
import { useRouter } from 'next/router';
import { useDispatch } from 'react-redux';
import styled from 'styled-components';
import { InterDataType } from '@/api/interface';
import { AppMallGoodsDetails } from '@/api/interface/mall';
import NumberBox from '@/components/numberBox';
import { setGlobalData } from '@/store/module/globalData';
import { formatMoney } from '@/utils/money';

// 商品详情类型
type DetailType = InterDataType<AppMallGoodsDetails>;

const SubmitProductView: React.FC<{ detail: DetailType }> = ({ detail }) => {
  // 路由钩子
  const router = useRouter();
  // store
  const dispatch = useDispatch();
  // 商品的购买数量
  const [productNum, setProductNum] = useState<number>(1);
  // 获取当前选择的规格
  const getCurrentSpec = () => {
    const { specId } = router.query;
    const item = detail?.priceStock?.find((i) => i?.id === Number(specId));
    // 找不到就用默认的
    return item || detail?.priceStock?.at(0);
  };
  // 转换规格展示
  const getCurrentSpecText = () => {
    const productSpec = getCurrentSpec()?.productSpec || undefined;
    if (!productSpec) return '请选择规格';
    return Object.entries(JSON.parse(productSpec))
      ?.map((i) => `【${i[0]}】${i[1]}`)
      ?.join(' +');
  };
  // 获取购买的价格
  const getCurrentPrice = () => {
    return Big(getCurrentSpec()?.salePrice || 1)
      .mul(Big(productNum))
      ?.toFixed(2)
      ?.toLocaleString();
  };
  // 设置购买数量
  const handleProductNum = (value: any) => {
    setProductNum(value);
    // 将购买数量存入store
    dispatch(
      setGlobalData({
        productSpecNum: value,
      }),
    );
  };
  // 组件挂载
  useEffect(() => {
    // 设置购买数量
    if (router?.query?.num) {
      setProductNum(Number(router?.query?.num));
      // 将购买数量存入store
      dispatch(
        setGlobalData({
          productSpecNum: Number(router?.query?.num),
        }),
      );
    }
  }, [router?.query]);
  return (
    <SubmitProductWrap>
      <div className="submit-title flex-between">
        <div className="title">确认订单信息</div>
      </div>
      <div className="submit-td flex-start">
        <div className="item">宝贝</div>
        <div className="item">单价</div>
        <div className="item">数量</div>
        <div className="item">合计</div>
      </div>
      <div className="submit-tr flex-start">
        <div className="item submit-product flex-start">
          <img
            src={detail?.resourcesList?.at(0)?.url}
            alt={detail?.tradeName}
            className="product-img"
          />
          <div className="product-content">
            <div className="title two-line-ellipsis">{detail?.tradeName}</div>
            <div className="desc">已选：{getCurrentSpecText()}</div>
          </div>
        </div>
        <div className="item">￥{formatMoney(getCurrentSpec()?.salePrice)}</div>
        <div className="item">
          <NumberBox
            min={1}
            max={9999}
            precision={0}
            value={productNum}
            onChange={handleProductNum}
          />
        </div>
        <div className="item submit-money">
          <span className="label">￥</span>
          <span className="num">{getCurrentPrice()}</span>
        </div>
      </div>
    </SubmitProductWrap>
  );
};

export default SubmitProductView;

// 样式
const SubmitProductWrap = styled.div`
  position: relative;
  width: 100%;
  box-sizing: border-box;
  padding-bottom: 1rem;
  .submit-title {
    position: relative;
    width: 100%;
    //border-bottom: 2px solid #f1f1f1;
    box-sizing: border-box;
    padding: 1rem 0 0.5rem 0;
    .title {
      font-size: 18px;
      font-weight: bold;
      color: #333333;
    }
    .action {
      font-weight: 400;
      color: #3366cc;
      cursor: pointer;
    }
  }
  .submit-td,
  .submit-tr {
    position: relative;
    width: 100%;
    min-height: 2rem;
    line-height: 2rem;
    border-bottom: 0.02rem solid #1f8afe;
    flex-wrap: nowrap;
    .item {
      text-align: center;
      color: #666666;
      font-weight: bold;
      &:nth-child(1) {
        width: 40%;
      }
      &:not(:nth-child(1)) {
        width: calc((100% - 40%) / 3);
      }
    }
  }
  .submit-tr {
    border: none;
    .item {
      display: flex;
      align-items: center;
      justify-content: center;
      min-height: 3.33rem;
      font-weight: normal;
    }
    .submit-product {
      width: 100%;
      justify-content: flex-start;
      box-sizing: border-box;
      padding: 0.67rem 0.5rem;
      .product-img {
        width: 3.33rem;
        height: 3.33rem;
      }
      .product-content {
        padding-left: 0.5rem;
        text-align: left;
        .title {
          font-weight: bold;
          color: #333333;
        }
        .desc {
          font-size: 12px;
          font-weight: 400;
          color: #666666;
          line-height: normal;
        }
      }
    }
    .submit-money {
      font-size: 18px;
      //line-height: 1;
      color: #ff6700;
      .label {
        font-size: 14px;
      }
    }
  }
`;
