import React, { useEffect, useState } from 'react';
import { useRouter } from 'next/router';
import { useDispatch } from 'react-redux';
import styled from 'styled-components';
import { HomeAPI } from '@/api';
import { InterDataType } from '@/api/interface';
import { RecommendGoodsType } from '@/api/interface/home';
import { setGlobalData } from '@/store/module/globalData';

// 列表类型
type ListType = InterDataType<RecommendGoodsType>;

const HomeProductView = () => {
  // 路由钩子
  const router = useRouter();
  // store
  const dispatch = useDispatch();
  // 推荐商品列表
  const [recommendGoodsList, setRecommendGoodsList] = useState<
    ListType[0]['mallGoodsList']
  >([]);
  // 获取推荐商品
  const getRecommendGoodsList = async () => {
    const res = await HomeAPI.getRecommendGoods();
    if (res && res.code === '200') {
      const list = res.result || [];
      setRecommendGoodsList(
        list
          ?.map((i) => i.mallGoodsList)
          ?.flat()
          ?.slice(0, 10),
      );
    }
  };
  // 获取最低价格
  const getLowerPrice = (item: ListType[0]['mallGoodsList'][0]) => {
    return (
      item.priceStock?.reduce((a: any, b: any) =>
        a.salePrice < b.salePrice ? a : b,
      ).salePrice || 0
    );
  };
  // 跳转详情
  const handleDetail = (item: ListType[0]['mallGoodsList'][0]) => {
    dispatch(
      setGlobalData({
        loadingSpinnerVisible: true,
      }),
    );
    router.push(`/mall/product/${item.id}`).then();
  };
  // 组件挂载
  useEffect(() => {
    getRecommendGoodsList().then();
  }, []);
  return (
    <HomeProductWrap>
      {recommendGoodsList?.map((i, j) => (
        <div
          className="product-item flex-start"
          key={j}
          onClick={() => handleDetail(i)}
        >
          <img
            className="item-image"
            src={`${i.resourcesList[0].url}?x-oss-process=image/quality,Q_50`}
            alt={i.tradeName}
          />
          <div className="item-title text-ellipsis">{i.tradeName}</div>
          <div className="item-price">
            <span className="label">￥</span>
            <span className="num">
              {i.priceShow ? getLowerPrice(i).toLocaleString() : '咨询报价'}
            </span>
          </div>
        </div>
      ))}
    </HomeProductWrap>
  );
};
export const HomeProductWrap = styled.div`
  width: 100%;
  //min-height: 8.71rem;
  box-sizing: border-box;
  //background: aquamarine;
  padding: 0.83rem 0 0.15rem 0;
  display: flex;
  justify-content: flex-start;
  align-items: center;
  flex-wrap: wrap;
  .product-item {
    position: relative;
    width: calc((100% - 0.83rem * 5) / 6);
    height: 14.46rem;
    margin-right: 0.83rem;
    margin-bottom: 0.83rem;
    border: 0.04rem solid #e3e3e3;
    flex-direction: column;
    box-sizing: border-box;
    padding: 0.67rem 0 0 0;
    background: #fff;
    &:hover,
    &:active {
      filter: brightness(0.95);
    }
    &:nth-child(6n) {
      margin-right: 0;
    }
    .item-image {
      width: 9rem;
      height: 9rem;
      object-fit: cover;
      margin-bottom: 0.5rem;
    }
    .item-title {
      position: relative;
      width: 100%;
      box-sizing: border-box;
      padding: 0 1.2rem;
      text-align: left;
      font-size: 13px;
      font-weight: bold;
      margin-bottom: 0.54rem;
    }
    .item-price {
      position: relative;
      width: 100%;
      box-sizing: border-box;
      padding: 0 1rem;
      text-align: left;
      font-size: 16px;
      font-weight: bold;
      color: #ff1b1b;
    }
  }
`;
export default HomeProductView;
