import React from 'react';
import { PlusOutlined } from '@ant-design/icons';
import { Button } from 'antd';
import { useRouter } from 'next/router';
import styled from 'styled-components';
import { HomeAPI } from '@/api';
import { InterListType } from '@/api/interface';
import { GetAppGambitListType } from '@/api/interface/home';
import LayoutView from '@/components/layout';

// 列表类型
type ListType = InterListType<GetAppGambitListType>;
// 帖子类型
const indexList = [
  { label: 1, value: 'bg-[#FF392B]' },
  { label: 2, value: 'bg-[#FE792B]' },
  { label: 3, value: 'bg-[#FEA32B]' },
  { label: 4, value: 'bg-[#FEA32B]' },
  { label: 5, value: 'bg-[#FEA32B]' },
];
// 每次加载页面都会执行
export async function getServerSideProps() {
  // 话题列表
  let topicList: ListType = [];
  // 获取话题列表
  const getListGambit = async () => {
    const res = await HomeAPI.getAppGambitList({
      pageNo: 1,
      pageSize: 999,
    });
    if (res && res.code === '200') {
      topicList = res.result.list || [];
    }
  };
  // 依次获取接口数据
  await (async () => {
    await getListGambit();
  })();
  return { props: { topicList } };
}

const ForumTopicPage: React.FC<{ topicList: ListType }> = ({ topicList }) => {
  // 路由钩子
  const router = useRouter();
  // 跳转详情
  const handleDetail = async (item: ListType[0]) => {
    await router.push(`/forum/detail/${item.id}`);
  };
  // useEffect(() => {
  //   console.log('组件挂载 ===>', topicList);
  // }, []);
  return (
    <LayoutView>
      <ForumTopicWrap>
        <div className="topic-bg"></div>
        <div className="topic-wrap">
          <div className="topic-head">
            <img
              className="title"
              src="https://file.iuav.com/file/sharefly-forum-topic-title.png"
              alt="热门话题"
            />
          </div>
          <div className="topic-list">
            <div className="list-title">
              <div className="total">{topicList?.length}条话题</div>
              <Button type="primary" icon={<PlusOutlined />}>
                发布动态
              </Button>
            </div>
            {topicList?.map((i, j) => (
              <div key={j} className="list-item list-none">
                <div
                  className={`item-index ${
                    indexList[j]?.value || 'bg-[#FEA32B]'
                  }`}
                >
                  {j + 1}
                </div>
                <img
                  className="item-image"
                  src={i?.gambitIcon || i?.gambitCover}
                  alt={i?.gambitName}
                />
                <div className="item-content">
                  <div
                    className="content-name two-line-ellipsis cursor-pointer"
                    onClick={() => handleDetail(i)}
                  >
                    {i.gambitName}
                  </div>
                  <div className="content-desc">
                    <span>{i?.discussionCount}个帖子</span>
                    <span>{i?.discussionCount}个讨论</span>
                  </div>
                </div>
                <Button
                  icon={<PlusOutlined />}
                  shape={'round'}
                  type="primary"
                  ghost
                >
                  加入
                </Button>
              </div>
            ))}
          </div>
        </div>
      </ForumTopicWrap>
    </LayoutView>
  );
};

export default ForumTopicPage;
// 样式
const ForumTopicWrap = styled.div`
  .topic-bg {
    position: absolute;
    top: -10px;
    left: 0;
    background-image: url('https://file.iuav.com/file/sharefly-forum-topic-bg.png');
    background-size: cover;
    height: 100vh;
    width: 100%;
    z-index: -1;
    opacity: 0.86;
  }
  .topic-wrap {
    position: relative;
    max-width: 1190px;
    box-sizing: border-box;
    padding: 2rem 0 0 0;
    margin: 0 auto;
    .topic-head {
      position: relative;
      width: 100%;
      display: flex;
      align-items: center;
      justify-content: space-between;
      margin-bottom: 1rem;
      .title {
        width: 12rem;
      }
    }
    .topic-list {
      position: relative;
      width: 100%;
      min-height: 20rem;
      background: #fff;
      border-radius: 1.86rem;
      box-sizing: border-box;
      padding: 1rem 1.5rem;
      margin-bottom: 2rem;
      .list-title {
        position: relative;
        width: 100%;
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 1rem;
      }
      .list-item {
        position: relative;
        width: 100%;
        min-height: 5rem;
        box-sizing: border-box;
        border-radius: 0.68rem;
        overflow: hidden;
        display: flex;
        align-items: center;
        flex-wrap: nowrap;
        justify-content: flex-start;
        &:not(:last-child) {
          margin-bottom: 2rem;
        }
        .item-index {
          position: absolute;
          top: 0;
          left: 0;
          width: 1.5rem;
          height: 1rem;
          color: #fff;
          text-align: center;
          line-height: 1rem;
          border-bottom-right-radius: 0.25rem;
        }
        .item-image {
          height: 5rem;
          width: 8rem;
          object-fit: cover;
          margin-right: 0.67rem;
        }
        .item-content {
          position: relative;
          width: 30rem;
          margin-right: 5rem;
          .content-name {
            width: 100%;
            font-size: 13px;
            font-weight: bold;
            &:hover {
              color: #ff392b;
            }
          }
          .content-desc {
            span:not(:last-child) {
              margin-right: 0.5rem;
            }
          }
        }
      }
    }
  }
`;
