import React, { useEffect, useState } from 'react';
import { RightOutlined } from '@ant-design/icons';
import { Pagination, Rate } from 'antd';
import styled from 'styled-components';
import { HomeAPI } from '@/api';
import { InterListType, InterReqType } from '@/api/interface';
import { AppListPilotType } from '@/api/interface/home';

// 列表类型
type ListType = InterListType<AppListPilotType>;
// 参数类型
type ReqType = InterReqType<AppListPilotType>;

const TabView05 = () => {
  // 飞手列表
  const [flyerList, setFlyerList] = useState<ListType>([]);
  // 分页数据
  const [pagination, setPagination] = useState({
    pageNo: 1,
    pageSize: 8,
    totalPage: 0,
    totalCount: 0,
  });
  // 获取飞手列表
  const getFlyerList = async (data?: ReqType) => {
    const res = await HomeAPI.appListPilot({
      // 审批通过
      auditStatus: 1,
      pageNo: pagination.pageNo,
      pageSize: pagination.pageSize,
      ...data,
    });
    if (res && res.code === '200') {
      const { list, pageNo, totalPage, totalCount } = res.result;
      setFlyerList(list);
      setPagination({
        ...pagination,
        pageNo,
        totalPage,
        totalCount,
      });
      // console.log('列表数据 --->', list, pageNo, totalPage);
    }
  };
  // 组件挂载
  useEffect(() => {
    getFlyerList().then();
  }, []);
  return (
    <TabViewWrap className={'animate__animated animate__fast animate__fadeIn'}>
      <div className="tab-title flex-between">
        <div className="title-name">飞手约单</div>
      </div>
      <div className="tab-list flex-start">
        {flyerList?.map((i, j) => (
          <div className="list-item list-none" key={j}>
            <div className="item-arrow">
              <RightOutlined style={{ fontSize: 10, color: '#A0A0A0' }} />
            </div>
            <div className="item-head flex-start">
              <img className="head-image" src={i?.userImg} alt={i?.userName} />
              <div className="head-content">
                <div className="content-title flex-start">
                  <div className="title-name">{i?.userName}</div>
                  <Rate
                    allowHalf
                    disabled
                    defaultValue={4}
                    style={{ fontSize: '12px', color: '#FE562F' }}
                  />
                </div>
                <div className="content-text flex-start">
                  <div className="text-item">{i?.residentCity}</div>
                  <div className="text-item">{i?.age}岁</div>
                  <div className="text-item">{i?.yearsOfWorking}年经验</div>
                </div>
                <div className="content-tag flex-start">
                  {i?.pilotAbility?.map((n, m) => (
                    <div className="tag-item" key={m}>
                      {n.abilityName}
                    </div>
                  ))}
                </div>
              </div>
            </div>
            <div className="item-foot flex-start">
              <div className="foot-state">空闲</div>
              <div
                className="foot-text text-ellipsis"
                title={i?.individualResume}
              >
                {i?.individualResume}
              </div>
            </div>
          </div>
        ))}
      </div>
      <div className="tab-foot flex-end">
        <Pagination
          size="small"
          current={pagination.pageNo}
          pageSize={pagination.pageSize}
          total={pagination.totalCount}
          onChange={async (e) => {
            await getFlyerList({ pageNo: e });
          }}
        />
      </div>
    </TabViewWrap>
  );
};

// 样式
const TabViewWrap = styled.div`
  position: relative;
  width: 100%;
  box-sizing: border-box;
  padding: 0.58rem;
  .tab-title {
    width: 100%;
    box-sizing: border-box;
    border-bottom: 0.02rem solid #ededed;
    padding-bottom: 0.58rem;
    margin-bottom: 0.42rem;
    display: none;
    .title-name {
      font-size: 1rem;
      font-weight: 550;
      color: #333333;
    }
  }
  .tab-list {
    position: relative;
    width: 100%;
    flex-wrap: wrap;
    .list-item {
      position: relative;
      box-sizing: border-box;
      width: 14.13rem;
      min-height: 5.54rem;
      background: #ffffff;
      border-radius: 0.25rem;
      border: 0.04rem solid #e3e3e3;
      margin: 0 0.85rem 0.85rem 0;
      padding: 0.83rem 0 0 0.83rem;
      &:active,
      &:hover {
        filter: brightness(0.95);
      }
      .item-arrow {
        position: absolute;
        top: 0.67rem;
        right: 0.67rem;
      }
      .item-head {
        flex-wrap: nowrap;
        .head-image {
          width: 3rem;
          height: 3rem;
          border-radius: 4px;
          margin-right: 0.38rem;
        }
        .head-content {
          position: relative;
          .content-title {
            align-items: baseline;
            margin-bottom: 0.2rem;
            .title-name {
              font-size: 14px;
              font-weight: 550;
              color: #333333;
              margin-right: 0.2rem;
            }
          }
          .content-text {
            margin-bottom: 0.2rem;
            .text-item {
              font-size: 12px;
              font-weight: 400;
              margin-right: 0.5rem;
              &:last-child {
                margin-right: 0;
              }
            }
          }
          .content-tag {
            flex-wrap: wrap;
            margin-bottom: 0.33rem;
            .tag-item {
              min-width: 2.42rem;
              height: 0.79rem;
              border-radius: 0.17rem;
              border: 0.04rem solid #fe562f;
              text-align: center;
              line-height: 0.79rem;
              font-size: 0.5rem;
              font-weight: 400;
              color: #fe562f;
              margin-right: 0.33rem;
            }
          }
          &::after {
            position: absolute;
            content: '';
            bottom: 0;
            left: 0;
            width: 8.21rem;
            height: 0.02rem;
            background: #dedede;
          }
        }
      }
      .item-foot {
        min-height: 1.42rem;
        box-sizing: border-box;
        padding: 0.33rem 0;
        .foot-state {
          position: relative;
          width: 3rem;
          text-align: center;
          margin-right: 0.33rem;
          font-size: 12px;
          color: #44d979;
          &::after {
            position: absolute;
            content: '';
            top: calc((100% - 0.25rem) / 2);
            left: 0;
            width: 0.25rem;
            height: 0.25rem;
            background: #44d979;
            border-radius: 50%;
          }
        }
        .foot-text {
          width: calc(100% - 3.33rem);
          font-size: 12px;
          font-weight: 400;
          color: #999999;
        }
      }
    }
  }
  .tab-foot {
    position: relative;
    width: 100%;
  }
`;
export default TabView05;
