import React, { useEffect, useState } from 'react';
import { Empty } from 'antd';
import styled from 'styled-components';
import { InterDataType, InterListType, InterReqType } from '@/api/interface';
import {
  QueryCurriculumInfoListType,
  SelectCurriculumClassifyType,
} from '@/api/interface/course';
import { CourseAPI } from '@/api/modules/course';
import BreadcrumbView from '@/components/breadcrumb';
import CategorySelectView, { CategoryType } from '@/components/categorySelect';
import CourseListItem from '@/components/courseListItem';
import LayoutView from '@/components/layout';
import ProductListView from '@/components/productList';

// 分类类型
type CategoryListType = Array<
  InterDataType<SelectCurriculumClassifyType>[0] & {
    children: InterDataType<SelectCurriculumClassifyType> | null;
  }
>;
// 列表类型
type ListType = InterListType<QueryCurriculumInfoListType>;
// 请求参数类型
type ReqType = InterReqType<QueryCurriculumInfoListType>;

const CourseView: React.FC<{ categoryList: CategoryListType }> = (props) => {
  // 分类列表
  const [categoryList, setCategoryList] = useState<CategoryType>([]);
  // 转换分类列表
  const getCategoryList = () => {
    setCategoryList(
      props.categoryList?.map((i) => ({
        value: i.oneCourseId,
        label: i.name,
        children: i.children?.map((n) => ({
          value: n.twoCourseId,
          label: n.name,
        })),
      })),
    );
  };
  // 分页数据
  const [pagination, setPagination] = useState({
    pageNo: 1,
    pageSize: 15,
    totalCount: 0,
  });
  // 列表数据
  const [dataList, setDataList] = useState<ListType>([]);
  // 获取课程列表
  const getDataList = async (data: ReqType) => {
    const res = await CourseAPI.getCourseVideoList({
      pageNo: pagination.pageNo,
      pageSize: pagination.pageSize,
      ...data,
    });
    if (res && res.code === '200') {
      console.log('获取商品列表 --->', res);
      const { list, totalCount, pageNo, pageSize } = res.result;
      setDataList(list || []);
      setPagination({
        ...pagination,
        totalCount,
        pageNo,
        pageSize,
      });
    }
  };
  // 翻页回调
  const handlePageChange = async (pageNo: number, pageSize: number) => {
    await getDataList({ pageNo, pageSize });
  };
  // 分类筛选
  const handleSelect = async (e: { main?: number; second?: number[] }) => {
    if (e.second?.length && e.second?.length > 1) {
      await getDataList({
        oneCourseId: e?.main,
        pageNo: 1,
      });
    } else {
      await getDataList({
        oneCourseId: e?.main,
        twoCourseId: e?.second?.[0],
        pageNo: 1,
      });
    }
  };
  // 组件挂载
  useEffect(() => {
    if (!props) return;
    getCategoryList();
  }, [props]);
  return (
    <LayoutView>
      <CourseViewWrap>
        {/* 面包屑 */}
        <BreadcrumbView />
        {/* 类型筛选 */}
        <CategorySelectView
          list={categoryList}
          isMultiple={false}
          allText={'全部课程'}
          onSelect={handleSelect}
        />
        {/* 课程列表 */}
        <ProductListView pagination={pagination} onChange={handlePageChange}>
          {dataList?.length ? (
            dataList?.map((i, j) => <CourseListItem key={j} detail={i} />)
          ) : (
            <div className="list-empty flex-center">
              <Empty />
            </div>
          )}
        </ProductListView>
      </CourseViewWrap>
    </LayoutView>
  );
};

export default CourseView;
// 样式
const CourseViewWrap = styled.div`
  position: relative;
  max-width: 1190px;
  box-sizing: border-box;
  padding: 2rem 0 0 0;
  margin: 0 auto;
`;
