import React from 'react';
import { useRouter } from 'next/router';
import { useDispatch, useSelector } from 'react-redux';
import styled from 'styled-components';
import { InterListType } from '@/api/interface';
import { GetListAPPCompanyInspectionPageType } from '@/api/interface/service';
import { GlobalDataState, setGlobalData } from '@/store/module/globalData';
import { getSellNumberRandom } from '@/utils';
import { formatMoney } from '@/utils/money';

// 详情类型
type DetailType = InterListType<GetListAPPCompanyInspectionPageType>[0];

const ServiceListItem: React.FC<{ detail: DetailType }> = ({ detail }) => {
  // store
  const dispatch = useDispatch();
  // 路由钩子
  const router = useRouter();
  // globalData
  const globalData = useSelector(
    (state: any) => state.globalData,
  ) as GlobalDataState;
  // 跳转商品详情
  const handleDetail = () => {
    dispatch(setGlobalData({ loadingSpinnerVisible: true }));
    router.push(`/service/detail/${detail?.id}`).then();
  };
  // 获取商品的单位
  const getPriceUnit = () => {
    const unit = globalData?.priceUnitList?.find(
      (i) => i.id === detail?.inspectionPriceUnitId,
    );
    return unit?.unitName || '次';
  };
  return (
    <ServiceListItemWrap>
      <div
        className="list-item cursor-pointer list-none"
        onClick={handleDetail}
      >
        <img
          className="image"
          src={`${detail?.inspectionFirstImg}?x-oss-process=image/quality,q_10`}
          alt={detail?.inspectionDTO?.inspectionName}
        />
        <div className="content">
          <div className="flex-between w-full">
            <div className="title two-line-ellipsis">
              {/* 【{i?.industryTypeDTO?.typeName}】 */}
              {detail?.inspectionDTO?.inspectionName}
            </div>
            <div className="nums">
              已接{getSellNumberRandom(detail?.id, 4)}单
            </div>
          </div>
          <div className="money">
            {detail?.price ? (
              <>
                <span className="label">￥</span>
                <span
                  className="num"
                  title={`${formatMoney(detail?.price)}元起每${getPriceUnit()}`}
                >
                  {formatMoney(detail?.price)}
                </span>
                <span
                  className="unit text-ellipsis"
                  title={`${formatMoney(detail?.price)}元起每${getPriceUnit()}`}
                >
                  起/{getPriceUnit()}
                </span>
              </>
            ) : (
              <span className="label">咨询报价</span>
            )}
          </div>
          <div className="store flex-start">
            <div className="icon">{detail?.companyName?.[0]}</div>
            <div className="text text-ellipsis" title={detail?.companyName}>
              {detail?.companyName}
            </div>
          </div>
        </div>
      </div>
    </ServiceListItemWrap>
  );
};

export default ServiceListItem;
const ServiceListItemWrap = styled.div`
  position: relative;
  width: calc((100% - 3.33rem) / 4);
  box-sizing: border-box;
  background: #ffffff;
  box-shadow: 0 0.22rem 0.56rem 0 rgba(56, 116, 205, 0.04);
  margin: 0 0 1.11rem 0;
  &:not(:nth-child(4n)) {
    margin-right: 1.11rem;
  }
  &:hover {
    filter: brightness(0.96);
  }
  .image {
    width: 100%;
    height: 11.58rem;
    object-fit: cover;
  }
  .content {
    position: relative;
    width: 100%;
    box-sizing: border-box;
    padding: 0 0.25rem 0.5rem 0.25rem;
    .title {
      font-weight: bold;
      font-size: 13px;
      color: #333333;
      max-width: 7rem;
    }
    .nums {
      font-weight: 400;
      font-size: 0.67rem;
      color: #999999;
    }
    .money {
      font-size: 16px;
      font-weight: 500;
      color: #ff1b1b;
      .label {
        font-size: 12px;
        font-weight: bold;
      }
      .num {
        max-width: 60%;
      }
      .unit {
        margin-left: 0.25rem;
        font-size: 10px;
        font-weight: 400;
      }
    }
    .store {
      .icon {
        width: 1rem;
        height: 1rem;
        border-radius: 50%;
        background: #cbcbcb;
        line-height: 1rem;
        text-align: center;
        color: #fff;
        font-size: 10px;
        margin-right: 0.25rem;
      }
      .text {
        color: #838383;
      }
    }
  }
`;
