import React, { useEffect } from 'react';
import { debounce } from 'lodash';

const InfiniteScrollList: React.FC<{
  children: React.ReactNode;
  bottomDistance?: number;
  onReachBottom?: () => void;
}> = ({ children, bottomDistance = 0, onReachBottom }) => {
  // 获取滚动距离
  const handleScroll = () => {
    // 获取页面的视口高度和内容高度
    const { scrollHeight, scrollTop } = document.documentElement;
    // 对于现代浏览器，通常推荐使用 window.innerHeight 和 window.scrollY 来获取视口高度和滚动位置
    const { innerHeight, scrollY } = window;
    // 修正为使用 window 对象的属性获取滚动距离
    const scrollTopValue = Math.max(
      scrollY,
      scrollTop,
      document.body.scrollTop,
    );

    // 判断是否滚动到了底部
    if (scrollTopValue + innerHeight >= scrollHeight - bottomDistance) {
      onReachBottom?.();
    }
  };

  // 使用防抖函数来减少频繁触发
  const handleDebounce = debounce(handleScroll, 100);

  // 使用 useEffect 钩子来添加和移除滚动事件的监听器
  useEffect(() => {
    // 组件挂载后，添加监听器
    window.addEventListener('scroll', handleDebounce);

    // 在组件卸载前，不仅要移除监听器，还需要清除防抖函数的定时器
    return () => {
      window.removeEventListener('scroll', handleDebounce);
      handleDebounce.cancel(); // Lodash debounce 提供了取消方法
    };
  }, []); // 依赖数组为空，表示只执行一次

  return <div>{children}</div>;
};

export default InfiniteScrollList;
