import React, { useEffect, useState } from 'react';
import { Form, Input, Modal, message, Select } from 'antd';
import { IndustryManageAPI } from '~/api';
import { IndustrySkuType } from '../../propsType';
import './index.scss';

interface PropsType {
  visible: boolean;
  closed: any;
  data: IndustrySkuType;
}

const { Option } = Select;

function AddOrEditIndustryModal(props: PropsType) {
  // 父组件传参
  const { visible, closed, data } = props;
  // 表格事件
  const [form] = Form.useForm();
  // 所属行业列表
  const [industryType, setIndustryType] = useState<{ label: string; val: number }[]>([]);
  // 关闭弹窗
  const handleClosed = async () => {
    form.setFieldsValue({
      goodsTypeId: undefined,
      description: undefined,
      solutionName: undefined,
      id: undefined,
    });
    closed();
  };
  // 获取所属行业列表
  const getFirstIndustryTypeInfo = async () => {
    const res: any = await IndustryManageAPI.getFirstIndustryTypeInfo({});
    if (res && res.code === '200') {
      const arr = res.result.map((i: { goodsMasterType: string; goodsMasterTypeId: number }) => {
        return {
          label: i.goodsMasterType,
          val: i.goodsMasterTypeId,
        };
      });
      setIndustryType(arr);
    }
  };
  // 触发表单验证
  const handleSubmit = () => {
    form
      .validateFields()
      .then(async (values) => {
        const obj = { ...data, ...values };
        const res: any = await IndustryManageAPI[!obj.id ? 'addIndustrySku' : 'editIndustrySku'](
          obj,
        );
        if (res && res.code === '200') {
          message.success('操作成功');
          await handleClosed();
        } else {
          message.warning(res.message);
        }
      })
      .catch((err) => {
        message.warning(err.errorFields[0].errors[0]).then();
      });
  };
  // componentsDidMounted
  useEffect(() => {
    if (!data) return;
    // console.log("新增编辑弹窗 -->", data);
    form.setFieldsValue({
      ...data,
    });
    getFirstIndustryTypeInfo().then();
  }, [data]);
  return (
    <Modal
      open={visible}
      title={data.id ? '修改' : '新增'}
      onCancel={handleClosed}
      onOk={handleSubmit}
      destroyOnClose
      width={768}
      wrapClassName='add-edit-modal'
    >
      <Form
        name='form'
        form={form}
        layout='horizontal'
        labelCol={{ span: 4 }}
        wrapperCol={{ span: 16 }}
      >
        <Form.Item
          name='solutionName'
          label='方案名称'
          rules={[{ required: true, message: '请输入方案名称' }]}
        >
          <Input placeholder='请输入方案名称' allowClear maxLength={30} />
        </Form.Item>
        <Form.Item
          name='goodsTypeId'
          label='所属行业'
          rules={[{ required: true, message: '请选择所属行业' }]}
        >
          <Select placeholder='请选择所属行业' allowClear>
            {industryType.map((i, j) => (
              <Option value={i.val} key={j}>
                {i.label}
              </Option>
            ))}
          </Select>
        </Form.Item>
        <Form.Item
          name='description'
          label='描述'
          rules={[{ required: false, message: '请输入方案描述' }]}
        >
          <Input.TextArea
            placeholder='请输入方案描述'
            allowClear
            maxLength={70}
            rows={4}
            showCount
          />
        </Form.Item>
      </Form>
    </Modal>
  );
}
export default AddOrEditIndustryModal;
