import React, { FC, useEffect } from 'react';
import { Form, Input, Modal, message, Select, ModalProps } from 'antd';
import { IndustryManageAPI } from '~/api';
import './index.scss';
import { InterDataType, InterReqType } from '~/api/interface';
import { categoryListType } from '~/api/interface/categoryManage';
import { industryAddType, listIndustryType } from '~/api/interface/industryManageType';

type categoryResponseType = InterDataType<categoryListType>['list'];
type industryListType = InterDataType<listIndustryType>['list'];
type industryFormType = InterReqType<industryAddType>;

interface PropsType {
  categoryIndustryList: categoryResponseType;
  industryItem: Partial<industryListType[0]> | undefined;
  close: () => void;
  addOrEditSuccess: () => void;
}

const AddOrEditIndustryModal: FC<ModalProps & PropsType> = ({
  open,
  categoryIndustryList,
  industryItem,
  close,
  addOrEditSuccess,
}) => {
  // 表格事件
  const [form] = Form.useForm<industryFormType>();

  // 关闭弹窗
  const handleClosed = () => {
    form.resetFields();
    close();
  };
  // 触发表单验证
  const handleSubmit = () => {
    form
      .validateFields()
      .then(async (values) => {
        const res: any = await IndustryManageAPI[
          industryItem ? 'editIndustryRequest' : 'addIndustryRequest'
        ]({ ...values, id: industryItem ? industryItem.id : undefined });
        if (res && res.code === '200') {
          message.success('操作成功');
          handleClosed();
          addOrEditSuccess();
        } else {
          message.warning(res.message);
        }
      })
      .catch((err) => {
        message.warning(err.errorFields[0].errors[0]).then();
      });
  };
  useEffect(() => {
    if (industryItem) {
      form.setFieldsValue({
        solutionName: industryItem.solutionName,
        categoryId: industryItem.categoryId,
        description: industryItem.description,
      });
    }
  }, [industryItem]);
  return (
    <Modal
      open={open}
      title={industryItem ? '修改' : '新增'}
      onCancel={handleClosed}
      onOk={handleSubmit}
      destroyOnClose
      width={768}
      wrapClassName='add-edit-modal'
    >
      <Form
        name='form'
        form={form}
        layout='horizontal'
        labelCol={{ span: 4 }}
        wrapperCol={{ span: 16 }}
      >
        <Form.Item
          name='solutionName'
          label='方案名称'
          rules={[{ required: true, message: '请输入方案名称' }]}
        >
          <Input placeholder='请输入方案名称' allowClear maxLength={30} />
        </Form.Item>
        <Form.Item
          name='categoryId'
          label='所属行业'
          rules={[{ required: true, message: '请选择所属行业' }]}
        >
          <Select placeholder='请选择所属行业' allowClear>
            {categoryIndustryList.map((i, j) => (
              <Select.Option value={i.id} key={j}>
                {i.classifyName}
              </Select.Option>
            ))}
          </Select>
        </Form.Item>
        <Form.Item
          name='description'
          label='描述'
          rules={[{ required: false, message: '请输入方案描述' }]}
        >
          <Input.TextArea
            placeholder='请输入方案描述'
            allowClear
            maxLength={70}
            rows={4}
            showCount
          />
        </Form.Item>
      </Form>
    </Modal>
  );
};
export default AddOrEditIndustryModal;
