import './index.scss';
import { Col, Form, Row, Select, Table } from 'antd';
import CommonSkuInfo from '~/components/goods/commonSkuInfo';
import { forwardRef, useEffect, useImperativeHandle, useRef, useState } from 'react';
import { RentManageAPI } from '~/api';
import { isEmptyBol, regPriceNumber } from '~/utils/validateUtils';
import { InterDataType, InterReqType } from '~/api/interface';
import { addRentGoodsType, leaseGoodsDetailsType } from '~/api/interface/rentManageType';
import { filterObjAttr } from '~/utils';

//租赁商品详情返回类型
type rentGoodsDetailType = InterDataType<leaseGoodsDetailsType>;
type EditableTableProps = Parameters<typeof Table>[0];
type ColumnTypes = Exclude<EditableTableProps['columns'], undefined>;

interface selfProps {
  ref: any;
  rentGoodsDetails: rentGoodsDetailType | undefined;
}
//规格表单数据类型
type specificationFormListType = {
  optionList: { label: string; value: string }[];
  id: number;
  name: string;
  addSpecificationValueShow: boolean;
  specificationValueList: { name: string; id: number; specificationName: string }[];
};
//sku返回类型
type priceStockType = (Exclude<InterReqType<addRentGoodsType>, undefined>['priceStock'][0] & {
  id: number;
})[];

const SkuInfo = forwardRef<any, selfProps>(({ rentGoodsDetails }, ref) => {
  const [form] = Form.useForm<{ minLeaseTerm: number; maxLeaseTerm: number }>();
  const commonSkuInfoRef = useRef<any>();

  //库存正则校验
  const stockValidator = (_rules: any, value: number) => {
    if (!isEmptyBol(value)) {
      if (/^[+]{0,1}(\d+)$/.test(value.toString())) {
        if (value > 99999999 || value < 0) {
          return Promise.reject(new Error('库存最大为99999999且大于0'));
        }
        return Promise.resolve();
      } else {
        return Promise.reject(new Error('请输入正整数'));
      }
    } else {
      return Promise.resolve();
    }
  };
  //价格正则校验
  const priceValidator = (_rules: any, value: number) => {
    if (!isEmptyBol(value)) {
      if (regPriceNumber(value.toString())) {
        if (value > 99999999 || value < 0) {
          return Promise.reject(new Error('价格最大为99999999且大于0'));
        }
        return Promise.resolve();
      } else {
        return Promise.reject(new Error('为整数且最多保留两位小数'));
      }
    } else {
      return Promise.reject(new Error('请输入金额'));
    }
  };
  const [defaultColumns, setDefaultColumns] = useState<
    (ColumnTypes[number] & {
      editable?: boolean;
      dataIndex?: string;
      inputType?: string;
      radioOption?: { name: string; id: number }[];
      rules?: any;
      maxLength?: number;
      children?: any;
      placeholder?: string;
    })[]
  >([
    {
      title: '商品规格',
      align: 'center',
      children: [],
      maxLength: 30,
    },
    {
      title: '缺货',
      align: 'center',
      editable: true,
      inputType: 'switch',
      dataIndex: 'stockOut',
    },
    {
      title: (
        <div>
          <span style={{ color: 'red' }}>*</span>
          <span>押金</span>
        </div>
      ),
      align: 'center',
      editable: true,
      dataIndex: 'cashPledge',
      width: '15%',
      placeholder: '押金',
      rules: [{ required: false, validator: priceValidator }],
    },
    {
      title: '库存',
      align: 'center',
      editable: true,
      dataIndex: 'stock',
      width: '10%',
      rules: [{ required: false, validator: stockValidator }],
    },
  ]);
  const [tableData, setTableData] = useState<priceStockType>([]);
  //全部租期下拉
  const [allLeaseTermInfoList, setAllLeaseTermInfoList] = useState<
    {
      label: string;
      value: number;
      key: string;
    }[]
  >([]);
  //  低租期下拉
  const [lowLeaseTermInfoList, setLowLeaseTermInfoList] = useState<
    {
      label: string;
      value: number;
      key: string;
    }[]
  >([]);
  //高租期下拉
  const [upLeaseTermInfoList, setUpLeaseTermInfoList] = useState<
    {
      label: string;
      value: number;
      key: string;
    }[]
  >([]);

  useImperativeHandle(ref, () => ({
    getForm: () => form,
    skuFormSubmit,
    getSpecificationValueForm: () => commonSkuInfoRef.current.getSpecificationValueForm(),
    getSpecificationForm: () => commonSkuInfoRef.current.getSpecificationForm(),
  }));

  const updateTableData = (tableData: any) => {
    setTableData([...tableData]);
  };
  const updateDefaultColumns = (columns: any) => {
    setDefaultColumns([...columns]);
  };

  //租赁-商品-租期信息
  const getLeaseTermInfo = () => {
    RentManageAPI.getLeaseTermInfo().then(({ result }) => {
      if (result) {
        const optionList = result.map((v, index) => ({
          label: v.leaseDate,
          value: v.id,
          key: [
            'threeDaysRental',
            'sevenDaysRental',
            'thirtyDaysRental',
            'ninetyDaysRental',
            'maxDaysRental',
          ][index],
        }));
        setLowLeaseTermInfoList(optionList);
        setUpLeaseTermInfoList(optionList);
        setAllLeaseTermInfoList(optionList);
      }
    });
  };
  //最低租期选择
  const lowLeaseTermInfoOnSelect = (value: number) => {
    const upLeaseTermInfoIndex = allLeaseTermInfoList.findIndex((v) => v.value === value);
    if (upLeaseTermInfoIndex !== -1) {
      setUpLeaseTermInfoList(
        allLeaseTermInfoList.slice(upLeaseTermInfoIndex, allLeaseTermInfoList.length),
      );
    }
    if (form.getFieldValue('maxLeaseTerm')) {
      const index = allLeaseTermInfoList.findIndex(
        (v) => v.value === form.getFieldValue('maxLeaseTerm'),
      );
      if (index !== -1) {
        const filterLeaseTermInfo = allLeaseTermInfoList.slice(upLeaseTermInfoIndex, index + 1);
        setDefaultColumns(
          defaultColumns
            .filter((v) => allLeaseTermInfoList.every((i) => i.key !== v.dataIndex))
            .concat(
              filterLeaseTermInfo.map((v) => ({
                title: (
                  <div>
                    <span style={{ color: 'red' }}>*</span>
                    <span>{v.label}</span>
                  </div>
                ),
                align: 'center',
                editable: true,
                dataIndex: v.key,
                width: '10%',
                placeholder: `${v.label}租金价格`,
                rules: [{ required: false, validator: priceValidator }],
              })),
            ),
        );
      }
    }
  };
  //  最高租期选择
  const upLeaseTermInfoOnSelect = (value: number) => {
    const lowLeaseTermInfoIndex = allLeaseTermInfoList.findIndex((v) => v.value === value);
    if (lowLeaseTermInfoIndex !== -1) {
      setLowLeaseTermInfoList(allLeaseTermInfoList.slice(0, lowLeaseTermInfoIndex + 1));
    }
    if (form.getFieldValue('minLeaseTerm')) {
      const index = allLeaseTermInfoList.findIndex(
        (v) => v.value === form.getFieldValue('minLeaseTerm'),
      );
      if (index !== -1) {
        const filterLeaseTermInfo = allLeaseTermInfoList.slice(index, lowLeaseTermInfoIndex + 1);
        setDefaultColumns(
          defaultColumns
            .filter((v) => allLeaseTermInfoList.every((i) => i.key !== v.dataIndex))
            .concat(
              filterLeaseTermInfo.map((v) => ({
                title: (
                  <div>
                    <span style={{ color: 'red' }}>*</span>
                    <span>{v.label}</span>
                  </div>
                ),
                align: 'center',
                editable: true,
                dataIndex: v.key,
                width: '10%',
                placeholder: `${v.label}租金价格`,
                rules: [{ required: false, validator: priceValidator }],
              })),
            ),
        );
      }
    }
  };

  const skuFormSubmit = async () => {
    try {
      const values: any[] = await Promise.all([
        form.validateFields(),
        commonSkuInfoRef.current.getSpecificationValueForm().validateFields(),
        commonSkuInfoRef.current.getSpecificationForm().validateFields(),
      ]);
      const specificationFormList: specificationFormListType[] =
        commonSkuInfoRef.current.getSpecificationFormList();
      const specificationFormItem = specificationFormList.find(
        (v) => !v.specificationValueList.length,
      );
      if (specificationFormItem) {
        return Promise.reject(`请为规格项${specificationFormItem.optionList[0].value}添加规格值`);
      } else {
        //规格项数据转化
        const specAttrList = specificationFormList.map((v) => ({
          specName: v.optionList[0].value,
          id: rentGoodsDetails
            ? rentGoodsDetails.specAttrList.find((i) => i.id === v.id)?.id
            : undefined,
          specValuesList: v.specificationValueList.map((j) => ({
            specName: j.name,
            id: rentGoodsDetails
              ? rentGoodsDetails.specAttrList
                  .find((i) => i.id === v.id)
                  ?.specValuesList.find((i) => i.id === j.id)?.id
              : undefined,
          })),
        }));
        //规格值数据转化
        const priceStock: priceStockType = tableData.reduce((pre: priceStockType, cur: any) => {
          //规格名，规格值组合类型
          const productSpec = specificationFormList.reduce(
            (a: { [x: string]: string }, b, currentIndex) => {
              a[b.optionList[0].value] = cur['name' + (currentIndex + 1)];
              return a;
            },
            {},
          );
          pre.push({
            ...Object.getOwnPropertyNames(values[1]).reduce((a: any, b) => {
              if (b.includes(cur.id)) {
                a[b.replace(cur.id, '')] = values[1][b];
              }
              return a;
            }, {}),
            productSpec: JSON.stringify(productSpec),
          });
          return pre;
        }, []);
        //获取租期内的最低价格
        const rentDateMinPrice = priceStock
          .reduce((pre: number[], cur: any) => {
            [
              'threeDaysRental',
              'sevenDaysRental',
              'thirtyDaysRental',
              'ninetyDaysRental',
              'maxDaysRental',
            ].map((key) => {
              if (cur[key]) {
                pre.push(Number(cur[key]));
              }
            });
            return pre;
          }, [])
          .sort((a, b) => a - b)[0];
        return Promise.resolve({
          ...values[0],
          priceStock,
          specAttrList,
          showPrice: rentDateMinPrice,
        });
      }
    } catch (error: any) {
      return Promise.reject(error);
    }
  };
  //表头拆分及合并列
  const mergeTableRow = (filterSpecificationFormList: specificationFormListType[]) => {
    const columns = filterSpecificationFormList.map((v, index) => ({
      title: v.optionList[0].value,
      dataIndex: 'name' + (index + 1),
      align: 'center',
      onCell: (_: any, i: number) => {
        //合并列
        if (index < filterSpecificationFormList.length - 1) {
          const count: number = filterSpecificationFormList
            .slice(index + 1, filterSpecificationFormList.length)
            .reduce((pre: number, cur) => {
              return pre * cur.specificationValueList.length;
            }, 1);
          return {
            rowSpan: count !== 1 ? ((i + 1) % count === 1 ? count : 0) : 1,
          };
        } else {
          return {
            rowSpan: 1,
          };
        }
      },
    }));
    defaultColumns[0].children = columns;
    setDefaultColumns([...defaultColumns]);
  };
  //排列组合规格值表单默认数据
  const setTableFormDefault = (tableDataList: any) => {
    const tableFormDefault = tableDataList.reduce((pre: any, cur: any) => {
      return {
        ...pre,
        ...Object.getOwnPropertyNames(filterObjAttr(cur, ['id'])).reduce((a: any, b) => {
          a[b + cur.id] = cur[b];
          return a;
        }, {}),
      };
    }, {});
    commonSkuInfoRef.current.getSpecificationValueForm().setFieldsValue(tableFormDefault);
  };

  useEffect(() => {
    getLeaseTermInfo();
  }, []);

  useEffect(() => {
    if (rentGoodsDetails) {
      form.setFieldsValue({
        minLeaseTerm: rentGoodsDetails.minLeaseTerm,
        maxLeaseTerm: rentGoodsDetails.maxLeaseTerm,
      });
      //转化数据
      const covertSpecAttrList = rentGoodsDetails.specAttrList.map((v, index) => ({
        id: v.id,
        name: 'specName' + index,
        optionList: [{ label: v.specName, value: v.specName }],
        specificationValueList: v.specValuesList.map((i) => ({
          id: i.id,
          name: i.specName,
          specificationName: v.specName,
        })),
        addSpecificationValueShow: false,
      }));
      //规格项表单数据默认数据
      const specFormDefault = rentGoodsDetails.specAttrList.reduce(
        (pre: any, cur: any, currentIndex) => {
          pre['specName' + currentIndex] = cur.specName;
          return pre;
        },
        {},
      );
      commonSkuInfoRef.current.getSpecificationForm().setFieldsValue(specFormDefault);
      commonSkuInfoRef.current.updateSpecificationFormList([...covertSpecAttrList]);
      const upLeaseTermInfoIndex = allLeaseTermInfoList.findIndex(
        (v) => v.value === rentGoodsDetails.maxLeaseTerm,
      );
      const lowLeaseTermInfoIndex = allLeaseTermInfoList.findIndex(
        (v) => v.value === rentGoodsDetails.minLeaseTerm,
      );
      const filterLeaseTermInfo = allLeaseTermInfoList.slice(
        lowLeaseTermInfoIndex,
        upLeaseTermInfoIndex + 1,
      );
      const addColumnsList: any = filterLeaseTermInfo.map((v) => ({
        title: (
          <div>
            <span style={{ color: 'red' }}>*</span>
            <span>{v.label}</span>
          </div>
        ),
        align: 'center',
        editable: true,
        dataIndex: v.key,
        width: '15%',
        placeholder: `${v.label}租金价格`,
        rules: [{ required: false, validator: priceValidator }],
      }));
      defaultColumns.push(...addColumnsList);
      mergeTableRow(covertSpecAttrList);

      const tableDataList: priceStockType = rentGoodsDetails.priceStock.map((v) => ({
        id: v.id,
        stockOut: !!v.stockOut,
        stock: v.stock || undefined,
        threeDaysRental: v.threeDaysRental || undefined,
        sevenDaysRental: v.sevenDaysRental || undefined,
        thirtyDaysRental: v.thirtyDaysRental || undefined,
        ninetyDaysRental: v.ninetyDaysRental || undefined,
        maxDaysRental: v.maxDaysRental || undefined,
        cashPledge: v.cashPledge,
        ...Object.getOwnPropertyNames(JSON.parse(v.productSpec)).reduce(
          (pre: any, cur, currentIndex) => {
            pre['name' + (currentIndex + 1)] = JSON.parse(v.productSpec)[cur];
            pre['specificationName' + (currentIndex + 1)] = cur;
            return pre;
          },
          {},
        ),
      }));
      setTableFormDefault(tableDataList);
      setTableData(tableDataList);
    }
  }, [rentGoodsDetails]);

  return (
    <div className='sku-info'>
      <div className='sku-info-title'>价格库存信息</div>
      <Row>
        <Col span={1}></Col>
        <Col span={7}>
          <Form wrapperCol={{ span: 18 }} labelCol={{ span: 5 }} form={form}>
            <Form.Item
              label='最低租期'
              name='minLeaseTerm'
              rules={[{ required: true, message: '请选择最低租期' }]}
            >
              <Select
                placeholder='请选择最低租期'
                options={lowLeaseTermInfoList}
                onSelect={lowLeaseTermInfoOnSelect}
              ></Select>
            </Form.Item>
            <Form.Item
              label='最高租期'
              name='maxLeaseTerm'
              rules={[{ required: true, message: '请选择最高租期' }]}
            >
              <Select
                placeholder='请选择最高租期'
                options={upLeaseTermInfoList}
                onSelect={upLeaseTermInfoOnSelect}
              ></Select>
            </Form.Item>
          </Form>
        </Col>
      </Row>
      <Row>
        <Col span={1}></Col>
        <Col span={22}>
          <CommonSkuInfo
            tableData={tableData}
            updateTableData={updateTableData}
            defaultColumns={defaultColumns}
            updateDefaultColumns={updateDefaultColumns}
            ref={commonSkuInfoRef}
          />
        </Col>
      </Row>
    </div>
  );
});
export default SkuInfo;
