import { useEffect, useState } from 'react';
import SearchBox from '~/components/search-box';
import { Button, message, Table } from 'antd';
import { ArrowLeftOutlined } from '@ant-design/icons';
import { activityInviteDetailType } from '~/api/interface/activityManage';
import { InterListType } from '~/api/interface';
import { ActivityManageAPI } from '~/api';
import { useLocation, useNavigate } from 'react-router-dom';
import qs from 'query-string';
import { ColumnsType } from 'antd/es/table';

// 列表的类型
type TableType = InterListType<activityInviteDetailType>;

const ActivityListDetailView = () => {
  // 路由钩子
  const location = useLocation();
  const navigate = useNavigate();
  // 表格数据
  const [tableData, setTableData] = useState<TableType>([]);
  // 表格分页配置
  const [pagination, setPagination] = useState({
    total: 0,
    pageSize: 10,
    current: 1,
    totalPage: 0,
  });
  // 活动id
  const activityId = Number(qs.parse(location.search).id);
  // 加载列表
  const getTableList = async (value = {}) => {
    // 只需要修改这个地方的接口即可
    const res = await ActivityManageAPI.activityInviteDetail({
      pageNo: pagination.current,
      pageSize: pagination.pageSize,
      activityId,
      ...value,
    });
    if (res && res.code === '200') {
      const { list, pageNo, totalCount, pageSize, totalPage } = res.result; // 解构
      setPagination({
        total: totalCount,
        current: pageNo,
        pageSize,
        totalPage,
      });
      setTableData(list || []);
      // console.log('加载列表 --->', list);
    } else {
      message.warning(res.message);
    }
  };
  // 翻页
  const paginationChange = (pageNo: number, pageSize: number) => {
    getTableList({ pageNo, pageSize }).then();
  };
  // 页面挂载
  useEffect(() => {
    getTableList().then();
  }, []);
  // 表格结构
  const columns: ColumnsType<TableType[0]> = [
    {
      title: '邀请人',
      dataIndex: 'userAccountId',
      align: 'center',
      render: (_text, record) =>
        record?.accountSimpleDTO?.userName || record?.accountSimpleDTO?.nickName,
    },
    {
      title: '邀请人手机号',
      dataIndex: 'userAccountId',
      align: 'center',
      render: (_text, record) => record?.accountSimpleDTO?.phoneNum,
    },
    {
      title: '被邀请人',
      dataIndex: 'userAccountId',
      align: 'center',
      render: (_text, record) =>
        record?.paccountSimpleDTO?.userName || record?.paccountSimpleDTO?.nickName,
    },
    {
      title: '被邀请人手机号',
      dataIndex: 'userAccountId',
      align: 'center',
      render: (_text, record) => record?.paccountSimpleDTO?.phoneNum,
    },
    {
      title: '奖励积分',
      dataIndex: 'rewardValue',
      align: 'center',
      render: (text) => `${text}分`,
    },
    {
      title: '注册时间',
      dataIndex: 'createTime',
      align: 'center',
    },
  ];
  return (
    <>
      <SearchBox
        child={
          <>
            <Button
              type={'primary'}
              icon={<ArrowLeftOutlined />}
              onClick={() => {
                navigate(-1);
              }}
            >
              返回
            </Button>
          </>
        }
      />
      <Table
        // size='small'
        dataSource={tableData}
        columns={columns}
        rowKey='id'
        // scroll={{ x: 1000 }}
        bordered
        pagination={{
          total: pagination.total,
          pageSize: pagination.pageSize,
          current: pagination.current,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
    </>
  );
};

export default ActivityListDetailView;
