import { FC, useEffect, useState } from 'react';
import { Form, Input, message, Modal, ModalProps } from 'antd';
import { Uploader } from '~/components/uploader';
import { UploadOutlined } from '@ant-design/icons';
import RichText from '~/components/richText';
import { InterDataType, InterReqType } from '~/api/interface';
import { addIndustryNewsType, industryNewsListType } from '~/api/interface/resourceManageType';
import { ResourceManageAPI } from '~/api';

//行业新闻返回类型
type industryNewsType = InterDataType<industryNewsListType>['list'];
//新增请求类型
type addNewsParameters = InterReqType<addIndustryNewsType>;
interface selfProps {
  handleOk: () => void;
  onCancel: () => void;
  currentIndustryNews?: industryNewsType[0];
}

const AddOrEditNewsModal: FC<ModalProps & selfProps> = ({
  open,
  onCancel,
  handleOk,
  currentIndustryNews,
}) => {
  const [form] = Form.useForm<addNewsParameters>();
  const [richTextValue, setRichTextValue] = useState<string>('');
  //封面图文件
  const [surfaceImgList, setSurfaceImgList] = useState<
    {
      id: number;
      name: string;
      uid: number;
      url: string;
    }[]
  >([]);
  const richTextChange = (html?: string) => {
    form.setFieldValue('newsContents', html || undefined);
  };
  //封面图-上传成功
  const uploadSuccess = (
    fileList: {
      id: number;
      name: string;
      uid: number;
      url: string;
    }[],
  ) => {
    setSurfaceImgList(fileList);
    form.setFieldValue('surfaceImg', fileList[0].url);
  };
  //提交-新增或编辑行业新闻
  const addOrEditModalOk = () => {
    form.validateFields().then((value: any) => {
      ResourceManageAPI[currentIndustryNews ? 'editIndustryNews' : 'addIndustryNews']({
        ...value,
        id: currentIndustryNews ? currentIndustryNews.id : undefined,
      }).then(({ code }) => {
        if (code === '200') {
          message.success(currentIndustryNews ? '编辑成功' : '新增成功');
          handleOk();
        }
      });
    });
  };
  const addOrEditModalCancel = () => {
    form.resetFields();
    setSurfaceImgList([]);
    setRichTextValue('');
    onCancel();
  };
  useEffect(() => {
    if (currentIndustryNews) {
      form.setFieldsValue({
        newsTitle: currentIndustryNews.newsTitle,
        origin: currentIndustryNews.origin,
        newsAuthor: currentIndustryNews.newsAuthor,
        surfaceImg: currentIndustryNews.surfaceImg,
        newsContents: currentIndustryNews.newsContents || undefined,
      });
      setSurfaceImgList([
        {
          id: Math.random(),
          name: 'surfaceImg',
          uid: Math.random(),
          url: currentIndustryNews.surfaceImg,
        },
      ]);
      setRichTextValue(currentIndustryNews.newsContents || '');
    }
  }, [currentIndustryNews]);
  return (
    <Modal
      open={open}
      onCancel={addOrEditModalCancel}
      onOk={addOrEditModalOk}
      title={currentIndustryNews ? '编辑' : '新建'}
      width={1000}
    >
      <Form labelCol={{ span: 4 }} wrapperCol={{ span: 16 }} form={form}>
        <Form.Item
          label='文章标题'
          name='newsTitle'
          rules={[{ required: true, message: '请输入文章标题' }]}
        >
          <Input placeholder='请输入文章标题' maxLength={30} />
        </Form.Item>
        <Form.Item
          label='文章来源'
          name='origin'
          rules={[{ required: true, message: '请输入文章来源' }]}
        >
          <Input placeholder='请输入文章来源' maxLength={30} />
        </Form.Item>
        <Form.Item
          label='作者'
          name='newsAuthor'
          rules={[{ required: true, message: '请输入文章作者' }]}
        >
          <Input placeholder='请输入文章作者' maxLength={30} />
        </Form.Item>
        <Form.Item
          label='封面图'
          name='surfaceImg'
          rules={[{ required: true, message: '请上传封面图' }]}
        >
          <Uploader
            fileUpload
            listType='picture-card'
            onChange={uploadSuccess}
            defaultFileList={surfaceImgList}
            fileType={['image/png', 'image/jpeg', 'image/jpg', 'image/gif', 'image/bmp']}
          >
            <UploadOutlined />
          </Uploader>
        </Form.Item>
        <Form.Item label='主题内容' name='newsContents'>
          <RichText onChange={richTextChange} value={richTextValue} videoSize={100} />
        </Form.Item>
      </Form>
    </Modal>
  );
};

export default AddOrEditNewsModal;
