import { useEffect, useState } from 'react';
import { useLocation, useNavigate } from 'react-router-dom';
import SearchBox from '~/components/search-box';
import { Button, Table, Tooltip } from 'antd';
import { ArrowLeftOutlined } from '@ant-design/icons';
import qs from 'query-string';
import { InterListType } from '~/api/interface';
import { releaseTenderNewsInfo } from '~/api/interface/resourceManageType';
import { ResourceManageAPI } from '~/api';
import { ColumnsType } from 'antd/es/table';
import AddEditDetailModal from '~/pages/resourceManage/tenderManage/comp/addEditDetailModal';

// 列表类型
type TableType = InterListType<releaseTenderNewsInfo>;

const TenderManageDetail = () => {
  // 路由钩子
  const location = useLocation();
  // 导航钩子
  const navigate = useNavigate();
  // 返回上一页
  const handleBack = () => {
    navigate(-1);
  };
  // 新增修改弹窗是否开启
  const [addModalVisible, setAddModalVisible] = useState(false);
  // 需要编辑的数据
  const [editData, setEditData] = useState<TableType[0]>();
  // 表格数据
  const [tableData, setTableData] = useState<TableType>([]);
  // 表格分页配置
  const [pagination, setPagination] = useState({
    total: 0,
    pageSize: 10,
    current: 1,
    totalPage: 0,
  });
  // 加载列表
  const getTableList = async (value = {}) => {
    // 只需要修改这个地方的接口即可
    const res = await ResourceManageAPI.releaseTenderNewsInfo({
      pageNo: pagination.current,
      pageSize: pagination.pageSize,
      ...value,
      tenderNewsId: Number(qs.parse(location.search)?.id),
    });
    if (res && res.code === '200') {
      const { list, pageNo, totalCount, pageSize, totalPage } = res.result; // 解构
      setPagination({
        total: totalCount,
        current: pageNo,
        pageSize,
        totalPage,
      });
      setTableData(list || []);
    }
  };
  // 翻页
  const paginationChange = (pageNo: number, pageSize: number) => {
    getTableList({ pageNo, pageSize }).then();
  };
  // componentDidMount
  useEffect(() => {
    getTableList().then();
  }, []);
  // 表格结构
  const columns: ColumnsType<TableType[0]> = [
    {
      title: '序号',
      dataIndex: 'accountNo',
      align: 'center',
      width: '50px',
      render: (_text, _record, index) => (pagination.current - 1) * pagination.pageSize + index + 1,
    },
    {
      title: '标题',
      dataIndex: 'tenderTitle',
      align: 'center',
      width: '200px',
      ellipsis: true,
    },
    {
      title: '详情',
      dataIndex: 'tenderContent',
      align: 'center',
      width: '300px',
      ellipsis: true,
      render: (text) =>
        text && (
          <Tooltip
            placement='top'
            title={
              <div
                dangerouslySetInnerHTML={{ __html: text.replace(/<img(?:(?!\/>).|\n)*?\/>/gm, '') }}
              />
            }
          >
            <div
              style={{
                height: '20px',
                overflow: 'hidden',
                textOverflow: 'ellipsis',
                whiteSpace: 'nowrap',
                wordBreak: 'break-all',
              }}
              dangerouslySetInnerHTML={{ __html: text }}
            />
          </Tooltip>
        ),
    },
    {
      title: '价格(万元)',
      dataIndex: 'tenderPrice',
      align: 'center',
      width: '100px',
      ellipsis: true,
      render: (text) => text?.toLocaleString(),
    },
    {
      title: '操作',
      dataIndex: 'action',
      align: 'center',
      width: '100px',
      fixed: 'right',
      render: (_text, record) => (
        <Button
          type={'link'}
          onClick={() => {
            setEditData(JSON.parse(JSON.stringify(record)));
            setAddModalVisible(true);
          }}
        >
          编辑
        </Button>
      ),
    },
  ];
  return (
    <>
      <SearchBox
        child={
          <>
            <Button type={'primary'} icon={<ArrowLeftOutlined />} onClick={handleBack}>
              返回
            </Button>
          </>
        }
      />
      <Table
        size='small'
        dataSource={tableData}
        columns={columns}
        rowKey='id'
        // scroll={{ x: 1500 }}
        bordered
        pagination={{
          total: pagination.total,
          pageSize: pagination.pageSize,
          current: pagination.current,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
      <AddEditDetailModal
        open={addModalVisible}
        title={editData?.id ? '编辑' : '新增'}
        data={editData}
        closed={() => {
          setAddModalVisible(false);
          setEditData(undefined);
          paginationChange(pagination.current, pagination.pageSize);
        }}
      />
    </>
  );
};

export default TenderManageDetail;
