import { Button, Image, message, Modal, Table } from 'antd';
import { ColumnsType } from 'antd/es/table';
import AddOrEditTypeModal from './components/addOrEditTypeModal';
import { useEffect, useState } from 'react';
import { InterDataType, PaginationProps } from '~/api/interface';
import { RentManageAPI } from '~/api';
import { getTypeListType } from '~/api/interface/rentManageType';

//类型类别返回类型
type rentTypeListType = InterDataType<getTypeListType>['list'];
const RentType = () => {
  const [addOrEditTypeModalShow, setAddOrEditTypeModalShow] = useState<boolean>(false);
  const [pagination, setPagination] = useState<PaginationProps & { totalCount: number }>({
    pageNo: 1,
    pageSize: 10,
    totalCount: 0,
  });

  const tableColumns: ColumnsType<rentTypeListType[0]> = [
    {
      title: '类型名称',
      align: 'center',
      dataIndex: 'name',
    },
    {
      title: '图片',
      align: 'center',
      dataIndex: 'url',
      render: (text: string) => <Image src={text} width={50} height={50} />,
    },
    {
      title: '操作',
      align: 'center',
      render: (_text: string, record) => (
        <>
          <Button type='link' onClick={() => addOrEditTypeModalClick(record)}>
            编辑
          </Button>
          <Button type='link' onClick={() => deleteRentType(record)}>
            删除
          </Button>
        </>
      ),
    },
  ];
  const [tableData, setTableData] = useState<rentTypeListType>([]);
  const [currentRentTypeItem, setCurrentRentTypeItem] = useState<rentTypeListType[0]>();

  //类型列表
  const getTypeList = () => {
    RentManageAPI.getTypeList({ pageNo: pagination.pageNo, pageSize: pagination.pageSize }).then(
      ({ result }) => {
        pagination.totalCount = result.totalCount;
        setPagination({ ...pagination });
        setTableData(result.list || []);
      },
    );
  };
  //新增，编辑类型
  const addOrEditTypeModalClick = (record?: rentTypeListType[0]) => {
    setCurrentRentTypeItem(record ? { ...record } : undefined);
    setAddOrEditTypeModalShow(true);
  };
  const addOrEditTypeModalCancel = () => {
    setAddOrEditTypeModalShow(false);
  };
  const addOrEditTypeModalOk = () => {
    setAddOrEditTypeModalShow(false);
    getTypeList();
  };
  //删除类型
  const deleteRentType = (record: rentTypeListType[0]) => {
    Modal.confirm({
      title: '提示',
      content: '确认删除该类型',
      onOk: () => {
        RentManageAPI.rentTypeRemove({ id: record.id }).then(({ code }) => {
          if (code === '200') {
            if (tableData.length === 1 && pagination.pageNo !== 1) {
              pagination.pageNo -= 1;
            }
            message.success('删除成功');
            getTypeList();
          }
        });
      },
    });
  };

  //分页
  const paginationChange = (pageNo: number, pageSize: number) => {
    pagination.pageNo = pageNo;
    pagination.pageSize = pageSize;
    getTypeList();
  };

  useEffect(() => {
    getTypeList();
  }, []);

  return (
    <div className='rent-type'>
      <div className='rent-type-operate' style={{ marginBottom: '10px' }}>
        <Button type='primary' onClick={() => addOrEditTypeModalClick()}>
          新增类型
        </Button>
      </div>
      <Table
        bordered
        columns={tableColumns}
        dataSource={tableData}
        rowKey='id'
        pagination={{
          total: pagination.totalCount,
          pageSize: pagination.pageSize,
          current: pagination.pageNo,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
      <AddOrEditTypeModal
        open={addOrEditTypeModalShow}
        onCancel={addOrEditTypeModalCancel}
        onOk={addOrEditTypeModalOk}
        currentRentTypeItem={currentRentTypeItem}
      />
    </div>
  );
};
export default RentType;
