import { Form, Input, message, Modal, ModalProps } from 'antd';
import { FC, useEffect, useState } from 'react';
import { Uploader } from '~/components/uploader';
import { UploadOutlined } from '@ant-design/icons';
import { InterDataType, InterReqType } from '~/api/interface';
import { addPrimaryCategoryType, categoryListRespType } from '~/api/interface/categoryManage';
import { CategoryManageAPI } from '~/api';

//分类列表返回类型
type categoryType = InterDataType<categoryListRespType>['list'];
//新增分类请求参数类型
type addPrimaryCategoryParameter = InterReqType<addPrimaryCategoryType>;
interface selfProps {
  onCancel: () => void;
  onOK: () => void;
  currentCategory: categoryType[0] | undefined;
}

const AddOrEditCategoryModal: FC<ModalProps & selfProps> = ({
  open,
  onCancel,
  onOK,
  currentCategory,
}) => {
  const [form] = Form.useForm<addPrimaryCategoryParameter>();
  const [fileList, setFileList] = useState<
    {
      id: number;
      name: string;
      uid: number;
      url: string;
    }[]
  >([]);
  //上传成功
  const uploadSuccess = (
    value: {
      id: number;
      name: string;
      uid: number;
      url: string;
    }[],
  ) => {
    form.setFieldValue('icon', value[0].url);
    setFileList([...value]);
  };
  //提交
  const handleOk = () => {
    form.validateFields().then((values: any) => {
      CategoryManageAPI[currentCategory ? 'updatePrimaryCategory' : 'addPrimaryCategory']({
        ...values,
        id: currentCategory ? currentCategory.id : undefined,
      }).then(({ code }) => {
        if (code === '200') {
          message.success('新增成功');
          form.resetFields();
          setFileList([]);
          onOK();
        }
      });
    });
  };
  const handleCancel = () => {
    form.resetFields();
    setFileList([]);
    onCancel();
  };
  useEffect(() => {
    if (currentCategory) {
      form.setFieldsValue({
        name: currentCategory.name,
        icon: currentCategory.icon,
        description: currentCategory.description || undefined,
      });
      setFileList([
        {
          name: 'categoryImg',
          id: Math.random(),
          uid: Math.random(),
          url: currentCategory.icon,
        },
      ]);
    }
  }, [currentCategory]);
  return (
    <Modal
      open={open}
      title={currentCategory ? '编辑分类' : '新增分类'}
      onCancel={handleCancel}
      onOk={handleOk}
    >
      <Form wrapperCol={{ span: 16 }} labelCol={{ span: 4 }} form={form}>
        <Form.Item
          label='分类名称'
          name='name'
          rules={[{ required: true, message: '请输入分类名称' }]}
        >
          <Input placeholder='请输入分类名称' maxLength={15} />
        </Form.Item>
        <Form.Item
          label='分类图标'
          name='icon'
          rules={[{ required: true, message: '请上传分类图标' }]}
        >
          <Uploader
            fileUpload
            listType='picture-card'
            onChange={uploadSuccess}
            fileLength={1}
            fileSize={5}
            defaultFileList={fileList}
          >
            <UploadOutlined />
          </Uploader>
        </Form.Item>
        <Form.Item label='分类描述' name='description'>
          <Input.TextArea placeholder='请输入分类描述' rows={4} maxLength={70} showCount />
        </Form.Item>
      </Form>
    </Modal>
  );
};
export default AddOrEditCategoryModal;
