import React, { useEffect } from 'react';
import { Form, Input, message, Modal } from 'antd';
import { InterListType, InterReqType } from '~/api/interface';
import {
  releaseTenderNewsInfoUpdate,
  releaseTenderNewsInfo,
} from '~/api/interface/resourceManageType';
import { ResourceManageAPI } from '~/api';
import RichText from '~/components/richText';
// 列表的类型
type TableType = InterListType<releaseTenderNewsInfo>;
// 请求的表单类型
type ReqType = InterReqType<releaseTenderNewsInfoUpdate>;
// 传参类型
interface propType {
  title: string;
  open: boolean;
  closed: any;
  data?: TableType[0];
}

const AddEditDetailModal: React.FC<propType> = (props) => {
  AddEditDetailModal.defaultProps = {
    data: undefined,
  };
  // 参数
  const { title, open, closed, data } = props;
  // 表单钩子
  const [form] = Form.useForm<ReqType>();
  // 关闭弹窗
  const handleCancel = () => {
    form.resetFields();
    closed();
  };
  // 确认事件
  const handleOk = () => {
    form
      .validateFields()
      .then(async (values) => {
        await handleSubmit(values as ReqType & { file: any[] });
      })
      .catch((err) => {
        message
          .warning({
            content: err.errorFields[0].errors[0],
          })
          .then();
      });
  };
  // 提交事件
  const handleSubmit = async (values: ReqType) => {
    const res = await ResourceManageAPI.releaseTenderNewsInfoUpdate({
      ...values,
      id: Number(data?.id),
    });
    if (res && res.code === '200') {
      message.success('操作成功');
      handleCancel();
    }
  };
  // componentDidMount
  useEffect(() => {
    if (!open) return;
    if (!data) return;
    form.setFieldsValue(data);
    // console.log('data --->', data);
  }, [open]);
  return (
    <Modal
      open={open}
      title={title}
      onCancel={handleCancel}
      onOk={handleOk}
      destroyOnClose
      width={768}
    >
      <Form
        name='Form'
        form={form}
        labelAlign='right'
        labelCol={{ span: 2 }}
        wrapperCol={{ span: 21 }}
      >
        <Form.Item
          label='标题'
          name='tenderTitle'
          rules={[{ required: true, message: '请输入标题' }]}
        >
          <Input placeholder={'请输入标题'} maxLength={50} allowClear />
        </Form.Item>
        <Form.Item
          label='价格'
          name='tenderPrice'
          rules={[
            { required: true, message: '请输入价格' },
            () => ({
              validator(_, value) {
                if (value && value.length > 10) {
                  return Promise.reject('金额不能超过十位数');
                }
                return Promise.resolve();
              },
            }),
          ]}
        >
          <Input
            type={'number'}
            placeholder={'请输入价格'}
            maxLength={10}
            allowClear
            suffix={'万元'}
          />
        </Form.Item>
        <Form.Item
          label='详情'
          name='tenderContent'
          rules={[{ required: true, message: '请输入详情' }]}
        >
          <RichText
            richTextContent={data?.tenderContent}
            onChange={(e) => form.setFieldValue('tenderContent', e)}
            height={250}
          />
        </Form.Item>
      </Form>
    </Modal>
  );
};

export default AddEditDetailModal;
