import React, { useEffect, useState } from 'react';
import { useLocation, useNavigate } from 'react-router-dom';
import qs from 'query-string';
import SearchBox from '~/components/search-box';
import { Button, Table } from 'antd';
import { ArrowLeftOutlined } from '@ant-design/icons';
import { InterListType } from '~/api/interface';
import { listUserRcdType } from '~/api/interface/customManageType';
import { CustomManageAPI } from '~/api';
import { ColumnsType } from 'antd/es/table';

// 表格类型
type TableType = InterListType<listUserRcdType>;

function CustomListDetail() {
  // 路由钩子
  const location = useLocation();
  // 导航钩子
  const navigate = useNavigate();
  // 表格数据
  const [tableData, setTableData] = useState<TableType>([]);
  // 表格分页配置
  const [pagination, setPagination] = useState({
    total: 0,
    pageSize: 5,
    current: 1,
    totalPage: 0,
  });
  // 加载列表
  const getTableList = async (value = {}) => {
    // 只需要修改这个地方的接口即可
    const res = await CustomManageAPI.listUserRcd({
      userAccountId: Number(qs.parse(location.search).id),
      pageNo: pagination.current,
      pageSize: pagination.pageSize,
      ...value,
    });
    if (res && res.code === '200') {
      const { list, pageNo, totalCount, pageSize, totalPage } = res.result; // 解构
      setPagination({
        total: totalCount,
        current: pageNo,
        pageSize,
        totalPage,
      });
      setTableData(list);
    }
  };
  // 翻页
  const paginationChange = (pageNo: number, pageSize: number) => {
    getTableList({ pageNo, pageSize }).then();
  };
  // 返回上一页
  const handleBack = () => {
    navigate(-1);
  };
  // componentDidMount
  useEffect(() => {
    getTableList().then();
  }, []);
  // 表格结构
  const columns: ColumnsType<TableType[0]> = [
    {
      title: '序号',
      dataIndex: 'accountNo',
      align: 'center',
      width: '50px',
      render: (_text, _record, index) => (pagination.current - 1) * pagination.pageSize + index + 1,
    },
    {
      title: '用户名称',
      dataIndex: 'userName',
      align: 'center',
      render: (text, record) => text || record.nickName || '游客用户',
    },
    {
      title: '手机号',
      dataIndex: 'phoneNum',
      align: 'center',
    },
    {
      title: '注册时间',
      dataIndex: 'createTime',
      align: 'center',
    },
  ];
  return (
    <>
      <SearchBox
        child={
          <Button type={'primary'} icon={<ArrowLeftOutlined />} onClick={() => handleBack()}>
            返回
          </Button>
        }
      />
      <Table
        size='small'
        dataSource={tableData}
        columns={columns}
        rowKey='id'
        // scroll={{ x: 1500 }}
        bordered
        pagination={{
          total: pagination.total,
          pageSize: pagination.pageSize,
          current: pagination.current,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
    </>
  );
}

export default CustomListDetail;
