package com.mmc.iuav.user.service.impl;

import com.mmc.iuav.page.PageResult;
import com.mmc.iuav.response.ResultBody;
import com.mmc.iuav.response.ResultEnum;
import com.mmc.iuav.user.client.PmsClient;
import com.mmc.iuav.user.dao.CompanyDao;
import com.mmc.iuav.user.entity.CompanyBackUserDO;
import com.mmc.iuav.user.entity.CompanyInfoDO;
import com.mmc.iuav.user.entity.CompanyMemberDO;
import com.mmc.iuav.user.entity.UserAccountDO;
import com.mmc.iuav.user.model.dto.LoginSuccessDTO;
import com.mmc.iuav.user.model.qo.CompanyInfoQO;
import com.mmc.iuav.user.model.vo.CompanyInfoVO;
import com.mmc.iuav.user.model.vo.UserAccountVO;
import com.mmc.iuav.user.service.CompanyService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author: zj
 * @Date: 2023/7/3 15:39
 */
@Service
public class CompanyServiceImpl implements CompanyService {

    @Autowired
    private CompanyDao companyDao;

    @Autowired
    private PmsClient pmsClient;

    @Override
    public ResultBody addCompanyInfo(CompanyInfoVO companyInfo) {
        int count = companyDao.countCompanyNames(companyInfo.getId(), companyInfo.getCompanyName());
        if (count > 0) {
            return ResultBody.error(ResultEnum.COMPANY_NAME_EXIST_ERROR);
        }
        CompanyInfoDO companyInfoDO = new CompanyInfoDO(companyInfo);
        companyDao.addCompanyInfo(companyInfoDO);
        return ResultBody.success();
    }

    @Override
    public ResultBody removeCompanyInfo(Integer id, String token) {
        // 查询绑定账号数量是否大于0
        int backAccountCount = companyDao.countCompanyBackUser(id);
        if (backAccountCount > 0) {
            return ResultBody.error(ResultEnum.REMOVE_COMPANY_BINDING_ERROR);
        }
        // 查询绑定服务数量是否大于0
        Integer taskCount = pmsClient.countCompanyTasks(id, token);
        if (taskCount == null || taskCount > 0) {
            return ResultBody.error(ResultEnum.REMOVE_COMPANY_TASK_ERROR);
        }
        // 有绑定小程序用户则不能删除
        List<CompanyMemberDO> memberDOS = companyDao.listCompanyMembers(id);
        if (!CollectionUtils.isEmpty(memberDOS)) {
            return ResultBody.error(ResultEnum.REMOVE_COMPANY_BINDING_APP_ERROR);
        }
        companyDao.removeCompanyInfo(id);
        return ResultBody.success();
    }

    @Override
    public ResultBody updateCompanyInfo(CompanyInfoVO companyInfo) {
        int count = companyDao.countCompanyNames(companyInfo.getId(), companyInfo.getCompanyName());
        if (count > 0) {
            return ResultBody.error(ResultEnum.COMPANY_NAME_EXIST_ERROR);
        }
        CompanyInfoDO companyInfoDO = new CompanyInfoDO(companyInfo);
        companyDao.updateCompanyInfo(companyInfoDO);
        return ResultBody.success();
    }

    @Override
    public ResultBody getCompanyInfoById(Integer id) {
        CompanyInfoDO companyInfoDO = companyDao.getCompanyInfoById(id);
        if (companyInfoDO != null) {
            return ResultBody.success(companyInfoDO.buildCompanyInfoVO());
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody<CompanyInfoVO> listCompanyPage(CompanyInfoQO param, LoginSuccessDTO loginSuccessDTO) {
        CompanyInfoDO companyInfoDO = companyDao.getCompanyInfoByBackUserAccountId(loginSuccessDTO.getUserAccountId());
        if (companyInfoDO != null && !companyInfoDO.getCompanyType().equals(0)) {
            param.setCompanyInfoId(companyInfoDO.getId());
        }
        int count = companyDao.countListCompanyPage(param);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(param.getPageNo(), param.getPageSize(), count));
        }
        Integer pageNo = param.getPageNo();
        param.buildCurrentPage();
        List<CompanyInfoDO> companyInfoDOS = companyDao.listCompanyPage(param);
        List<CompanyInfoVO> companyInfoVOList = companyInfoDOS.stream().map(CompanyInfoDO::buildCompanyInfoVO).collect(Collectors.toList());
        return ResultBody.success(PageResult.buildPage(pageNo, param.getPageSize(), count, companyInfoVOList));
    }

    @Override
    public CompanyInfoVO getCompanyInfoByBackUserAccountId(Integer backUserAccountId) {
        CompanyInfoDO companyInfoDO = companyDao.getCompanyInfoByBackUserAccountId(backUserAccountId);
        if (companyInfoDO != null) {
            return companyInfoDO.buildCompanyInfoVO();
        }
        return null;
    }

    @Override
    public void addCompanyBackUser(CompanyBackUserDO companyBackUserDO) {
        companyDao.addCompanyBackUser(companyBackUserDO);
    }

    @Override
    public void updateCompanyBackUser(CompanyBackUserDO companyBackUserDO) {
        companyDao.updateCompanyBackUser(companyBackUserDO);
    }

    @Override
    public List<CompanyInfoVO> listCompanyPageBack(CompanyInfoQO param) {
        Integer pageNo = param.getPageNo();
        param.buildCurrentPage();
        List<CompanyInfoDO> companyInfoDOS = companyDao.listCompanyPage(param);
        List<CompanyInfoVO> companyInfoVOList = companyInfoDOS.stream().map(CompanyInfoDO::buildCompanyInfoVO).collect(Collectors.toList());
        return companyInfoVOList;
    }

    @Override
    public List<CompanyInfoDO> listCompanyInfoByUID(List<Integer> backUserIds) {
        List<CompanyInfoDO> companyInfoDOList = companyDao.listCompanyInfoByUID(backUserIds);
        return companyInfoDOList;
    }

    @Override
    public ResultBody bindingCompanyMember(LoginSuccessDTO loginSuccessDTO, Integer userAccountId, Integer companyInfoId) {
        // 查看该成员是否已被绑定
        int count = companyDao.countBingingCompanyInfoByUID(userAccountId);
        if (count > 0) {
            return ResultBody.error(ResultEnum.REBINDING_COMPANY_ERROR);
        }
        CompanyMemberDO companyMemberDO = new CompanyMemberDO();
        companyMemberDO.setUserAccountId(userAccountId);
        companyMemberDO.setCompanyInfoId(companyInfoId);
        companyMemberDO.setLeader(0);
        companyDao.addCompanyMember(companyMemberDO);
        return ResultBody.success();
    }

    @Override
    public CompanyInfoVO getAppCompanyInfoUId(Integer userAccountId) {
        CompanyInfoDO companyInfoDO = companyDao.getAppCompanyInfoUId(userAccountId);
        if (companyInfoDO != null) {
            return companyInfoDO.buildCompanyInfoVO();
        }
        return null;
    }

    @Override
    public CompanyBackUserDO getCompanyBackUserByCompanyId(Integer id) {
        CompanyBackUserDO companyBackUserDO = companyDao.getCompanyBackUserByCompanyId(id);
        return companyBackUserDO;
    }

    @Override
    public ResultBody unbindCompanyMember(LoginSuccessDTO userLoginInfoFromRedis, Integer userAccountId, Integer companyInfoId) {
        CompanyMemberDO companyMemberDO = companyDao.getCompanyMember(userAccountId, companyInfoId);
        if (companyMemberDO != null) {
            if (companyMemberDO.getLeader().equals(1)) {
                return ResultBody.error(ResultEnum.UNBIND_COMPANY_MEMBER_ERROR);
            }
        }
        companyDao.unbindCompanyMember(userAccountId, companyInfoId);
        return ResultBody.success();
    }

    @Override
    public ResultBody transferLeader(LoginSuccessDTO loginSuccessDTO, Integer fromUserAccountId, Integer toUserAccountId, Integer companyInfoId) {
        CompanyMemberDO companyMemberDO = companyDao.getCompanyMember(fromUserAccountId, companyInfoId);
        if (companyMemberDO != null) {
            if (!companyMemberDO.getLeader().equals(1)) {
                return ResultBody.error(ResultEnum.TRANFER_COMPANY_MEMBER_LEADER_ERROR);
            }
        }
        companyDao.updateCompanyMember(fromUserAccountId, 0, companyInfoId);
        companyDao.updateCompanyMember(toUserAccountId, 1, companyInfoId);
        return ResultBody.success();
    }

    @Override
    public ResultBody listCompanyMembers(Integer companyInfoId, LoginSuccessDTO loginSuccessDTO, Integer pageNo, Integer pageSize) {
        int count = companyDao.countCompanyMembers(companyInfoId);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(pageNo, pageSize, count));
        }
        pageNo = (pageNo - 1) * pageSize;
        List<UserAccountDO> userAccountDOS = companyDao.listCompanyMembersPage(companyInfoId, pageNo, pageSize);
        List<UserAccountVO> userAccountVOS = userAccountDOS.stream().map(UserAccountDO::buildUserAccountVO).collect(Collectors.toList());
        return ResultBody.success(PageResult.buildPage(pageNo, pageSize, count, userAccountVOS));
    }
}
